/**
 * @brief Initialize EOS sound system to use SmartKeys sounds.
 * @note this must be called before any smartkeys_sound_play() commands.
 */

#include <msx.h>
#include <eos.h>
#include <intrinsic.h>
#include <interrupt.h>
#include <arch/z80.h>
#include <stdbool.h>

/* Sound data pulled from the SmartWriter ROM */

static unsigned char output1[10], output2[10], output3[10], output4[10], output5[10], output6[10];

static unsigned char sound0[] = {0x40,0x87,0x50,0x03,0x42,0x8F,0x30,0x0A,0x17,0x11,0x50};                                                   // 0E74 [0]
static unsigned char sound1[] = {0x80,0xD6,0x50,0x03,0x82,0xE2,0x30,0x0A,0x17,0x11,0x90};                                                   // 0E7F [1]
static unsigned char sound2[] = {0x02,0x63,0x0A,0x00,0x00,0x10};                                                                            // 0E8A [2]
static unsigned char sound3[] = {0xC0,0x4C,0xF0,0x0A,0xD0};                                                                                 // 0E90 [3]
static unsigned char sound4[] = {0x02,0x67,0x02,0x00,0x00,0x10};                                                                            // 0E95 [4]
static unsigned char sound5[] = {0xC0,0x0A,0xF0,0x02,0xD0};                                                                                 // 0E9B [5]
static unsigned char sound6[] = {0x41,0x40,0x60,0x02,0x55,0x15,0x40,0x80,0x60,0x05,0x50,0x81,0x57,0x63,0x02,0x44,0xa3,0x90};                // 0EA0 [6]
static unsigned char sound7[] = {0x41,0x40,0x40,0x02,0x33,0xFC,0x41,0x40,0x40,0x02,0x33,0xfc,0x42,0x40,0x40,0x09,0x1d,0x11,0x50};           // 0EB2 [7]
static unsigned char sound8[] = {0x81,0x4C,0x20,0x02,0x33,0xFB,0x81,0x4C,0x81,0x4C,0x20,0x02,0x33,0xFB,0x82,0x4C,0x20,0x09,0x1D,0x11,0x90}; // 0EC5 [8]
static unsigned char sound9[] = {0xC1,0x60,0x20,0x02,0x33,0xFA,0xC1,0x60,0x20,0x02,0x33,0xFA,0xC2,0x60,0x20,0x09,0x1D,0x11,0xD0};          // 0ED8 [9]
static unsigned char sound10[] = {0x42,0x20,0x40,0x10,0x24,0x14,0x42,0x20,0x30,0x14,0x24,0x16,0x50};                                        // 0EEB [10]
static unsigned char sound11[] = {0x82,0x40,0x40,0x10,0x18,0x13,0x82,0x40,0x30,0x14,0x18,0x15,0x90};                                        // 0EF8 [11]
static unsigned char sound12[] = {0xC2,0x80,0x40,0x10,0x18,0x12,0xC2,0x80,0x30,0x14,0x18,0x11,0xD0};                                        // 0F05 [12]
static unsigned char sound13[] = {0x82,0x40,0x40,0x0F,0x18,0x15,0x90,0x42,0x20,0x30,0x14,0x26,0x1B,0x50,0x82,0x40,0x30,0x14,0x1B,0x19,0x90}; // 0F12 [13]
static unsigned char sound14[] = {0x82,0xD6,0x70,0x40,0x32,0x12,0x90};                                                                                 // 0F2E [14]
static unsigned char sound15[] = {0x82,0xCA,0x70,0x04,0x32,0x12,0x90};                                                                       // 0F35 [15]
static unsigned char sound16[] = {
0xc0,0xdd,0xb2,1,
0xc0,0xdb,0xb2,1,
0xc0,0xd7,0xb2,1,
0xc0,0xd3,0xb2,1,
0xc0,0xcf,0xb2,1,
0xc0,0xc3,0xb2,1,
0xc0,0xbb,0xb2,1,
0xc0,0xad,0xa2,1,
0xc0,0x99,0xa2,1,
0xc0,0x85,0xa2,1,
0xc0,0x6d,0x92,1,
0xc0,0x4d,0x92,1,
0xc0,0x2b,0x92,1,
0xc0,0xfb,0x81,1,
0xc0,0xd3,0x81,1,
0xc0,0xaf,0x71,1,
0xc0,0x91,0x71,1,
0xc0,0x73,0x61,1,
0xc0,0x59,0x51,1,
0xc0,0x41,0x51,1,
0xc0,0x29,0x51,1,
0xc0,0x17,0x41,1,
0xc0,0x07,0x41,1,
0xc0,0xf7,0x40,1,
0xc0,0xeb,0x40,1,
0xc0,0xdd,0x40,1,
0xc0,0xd3,0x50,1,
0xc0,0xc9,0x60,1,
0xc0,0xc1,0x70,1,
0xc0,0xbb,0x80,1,
0xd0
};
static unsigned char sound17[] = {0x40,0x3d,0x31,2,
0x40,0x57,0x32,1,
0x40,0x19,0x33,1,
0x40,0x15,0x33,1,
0x40,0x5f,0x33,1,
0x40,0x63,0x33,1,
0x40,0xbd,0x33,1,
0x40,0xc1,0x33,1,
0x40,0xeb,0x33,1,
0x40,0x39,0x31,1,
0x40,0x15,0x32,1,
0x40,0xa9,0x32,1,
0x40,0x3d,0x33,1,
0x40,0xad,0x33,1,
0x40,0xf5,0x33,2,
0x40,0xf7,0x33,1,
0x40,0x1b,0x32,1,
0x40,0xb9,0x32,1,
0x40,0x21,0x33,1,
0x40,0xb1,0x33,1,
0x40,0xff,0x33,1,
0x40,0xf7,0x33,3,
0x40,0xf5,0x33,1,
0x40,0x2f,0x32,1,
0x40,0x4d,0x33,1,
0x40,0x9d,0x31,1,
0x40,0x3b,0x32,1,
0x40,0x8b,0x33,1,
0x40,0xf7,0x33,1,
0x40,0xf3,0x33,1,
0x40,0xf1,0x33,1,
0x40,0xed,0x33,1,
0x40,0xf3,0x33,1,
0x40,0xf7,0x33,2,
0x40,0xf7,0x33,1,
0x40,0xfb,0x33,1,
0x50
};

static unsigned char sound18[] = { // wellerman
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xA7,0x32,15,0x1c,0x11, //E-3
0x42,0x5D,0x32,15,0x1c,0x11, //F#3
0x42,0x3B,0x32,15,0x1c,0x11, //G-3
0x42,0xFC,0x31,15,0x1c,0x11, //A-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0x53,0x31,15,0x1c,0x11, //E-4
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xA7,0x32,15,0x1c,0x11, //E-3
0x42,0x5D,0x32,15,0x1c,0x11, //F#3
0x42,0x3B,0x32,15,0x1c,0x11, //G-3
0x42,0xFC,0x31,15,0x1c,0x11, //A-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xC5,0x31,15,0x1c,0x11, //B-3
0x42,0xFC,0x31,15,0x1c,0x11, //A-3
0x42,0x3B,0x32,15,0x1c,0x11, //G-3
0x42,0x3B,0x32,15,0x1c,0x11, //G-3
0x42,0x5D,0x32,15,0x1c,0x11, //F#3
0x42,0xA7,0x32,15,0x1c,0x11, //E-3
0x90
};

static unsigned char sound19[] = { // dice roll - p1
  0x00,0x00,0x33,120,
  0x10
};
static unsigned char sound20[] = { // dice roll - p2
0xc0,0xff,0xf0,1,
0xc0,0x1b,0xf1,1,
0xc0,0x33,0xf1,1,
0xc0,0xe5,0xf2,1,
0xc0,0x0f,0xf3,1,
0xc0,0x31,0xf3,1,
0xc0,0x73,0xf3,1,
0xc0,0x7d,0xf3,1,
0xc0,0x5f,0xf2,1,
0xc0,0xb1,0xf3,1,
0xc0,0xd3,0xf3,1,
0xc0,0xe7,0xf3,4,
0xc0,0xff,0xf0,1,
0xc0,0xa3,0xf1,1,
0xc0,0xa9,0xf2,1,
0xc0,0xd7,0xf3,2,
0xc0,0xb7,0xf3,1,
0xc0,0xeb,0xf3,1,
0xc0,0xed,0xf3,1,
0xc0,0x07,0xf1,1,
0xc0,0x4d,0xf1,1,
0xc0,0xcb,0xf1,1,
0xc0,0xc3,0xf1,1,
0xc0,0xf3,0xf3,1,
0xc0,0xf7,0xf3,1,
0xc0,0xf9,0xf3,1,
0xc0,0xf7,0xf3,2,
0xc0,0xf5,0xf3,2,
0xc0,0xd9,0xf1,1,
0xc0,0x2f,0xf1,1,
0xc0,0x39,0xf2,1,
0xc0,0x2f,0xf3,1,
0xc0,0x9f,0xf3,1,
0xc0,0xb5,0xf2,1,
0xc0,0xd5,0xf3,1,
0xc0,0xe3,0xf3,1,
0xc0,0xf3,0xf3,1,
0xc0,0xf1,0xf3,1,
0xc0,0xf7,0xf3,11,
0xc0,0x9d,0xf1,1,
0xc0,0x43,0xf3,1,
0xc0,0x5f,0xf3,1,
0xc0,0x59,0xf3,1,
0xc0,0x03,0xf3,1,
0xc0,0x21,0xf3,1,
0xc0,0xfb,0xf3,8,
0xc0,0xfd,0xf3,1,
0xc0,0x69,0xf3,1,
0xc0,0x4d,0xf3,1,
0xc0,0xab,0xf3,1,
0xc0,0xff,0xf3,14,
0xc0,0xfd,0xf3,1,
0xc0,0xfb,0xf3,2,
0xc0,0xf7,0xf3,1,
0xc0,0xf3,0xf3,1,
0xc0,0xef,0xf3,2,
0xc0,0x27,0xf0,1,
0xc0,0x71,0xf0,1,
0xc0,0xeb,0xf3,1,
0xc0,0xe7,0xf3,1,
0xc0,0xe5,0xf3,1,
0xc0,0x7d,0xf0,1,
0xc0,0x7b,0xf0,1,
0xc0,0xe1,0xf3,1,
0xc0,0xd1,0xf3,1,
0xc0,0xcd,0xf3,1,
0xc0,0x81,0xf0,1,
0xc0,0x79,0xf0,1,
0xc0,0xd7,0xf3,1,
0xc0,0xc1,0xf3,1,
0xc0,0xb7,0xf3,1,
0xc0,0x73,0xf0,1,
0xc0,0x79,0xf0,1,
0xc0,0x91,0xf3,1,
0xc0,0xff,0xf0,1,
0xc0,0x6d,0xf0,1,
0xc0,0x6f,0xf0,1,
0xc0,0xe3,0xf3,1,
0xc0,0xeb,0xf3,1,
0xc0,0xed,0xf3,1,
0xc0,0x71,0xf0,1,
0xc0,0x69,0xf0,1,
0xd0

};

static unsigned short soundTable[42];

static void nmi_play(void)
{
  M_PRESERVE_ALL;
  eos_play_sound();
  M_RESTORE_ALL;
}

static void *table;
static void *nextnote;
static void *end;

void sound_init(void)
{
  Z80_registers r;
    
  r.Bytes.B = 6; // 6 output tables
  r.UWords.HL = &soundTable;

  // 16 sound fragments
  
  soundTable[0]=&sound10; soundTable[1]=&output1;
  soundTable[2]=&sound11; soundTable[3]=&output2;
  soundTable[4]=&sound12; soundTable[5]=&output3;
  soundTable[6]=&sound0; soundTable[7]=&output1;
  soundTable[8]=&sound1; soundTable[9]=&output2;
  soundTable[10]=&sound7; soundTable[11]=&output4;
  soundTable[12]=&sound8; soundTable[13]=&output5;
  soundTable[14]=&sound9; soundTable[15]=&output6;
  soundTable[16]=&sound2; soundTable[17]=&output4;
  soundTable[18]=&sound3; soundTable[19]=&output5;
  soundTable[20]=&sound4; soundTable[21]=&output1;
  soundTable[22]=&sound5; soundTable[23]=&output2;
  soundTable[24]=&sound6; soundTable[25]=&output2;
  soundTable[26]=&sound14; soundTable[27]=&output1;
  soundTable[28]=&sound15; soundTable[29]=&output1;
  soundTable[30]=&sound13; soundTable[31]=&output2;
  soundTable[32]=&sound16; soundTable[33]=&output2;
  soundTable[34]=&sound17; soundTable[35]=&output1;
  soundTable[36]=&sound18; soundTable[37]=&output1;
  soundTable[38]=&sound19; soundTable[39]=&output4;
  soundTable[40]=&sound20; soundTable[41]=&output2;
  
  AsmCall(0xFD50,&r,REGS_ALL,REGS_ALL);
  add_raster_int(nmi_play);
}

static void _play(unsigned char n)
{  
  eos_start_sound(n,end,nextnote,table);
}

void main(void) {

sound_init();

  while(true)
    {

    unsigned char k = eos_read_keyboard();
    switch(k)
      {
      case 'a':
        _play(11);
        _play(12);
        break;
      case 's':
        _play(13);
        break;
      case 'd':
        _play(16);
        break;
      case 'f':
        _play(6);
        _play(7);
        _play(8);
        break;
      case 'g':
        _play(1);
        _play(2);
        _play(3);
        break;
      case 'h':
        _play(14);
        break;
      case 'j':
        _play(15);
        break;
      case 'k':
        _play(9);
        _play(10);
        break;
      case 'l':
        _play(4);
        _play(5);
        break;
      case 'z':        
        _play(3);
        break;
      case 'x':        
        _play(17);
        break;
      case 'c':        
        _play(18);
        break;
      case 'v':        
        _play(19);
        break;
      case 'b':
        _play(20);
        _play(21);
        break;
      default:
        break;
      }
    }

}