/** ADAMEm: Coleco ADAM emulator ********************************************/
/**                                                                        **/
/**                               pp2bmp.c                                 **/
/**                                                                        **/
/** This program converts powerpaint binaries to Windows bitmap format     **/
/**                                                                        **/
/** Copyright (C) Marcel de Kogel 1997                                     **/
/**     You are not allowed to distribute this software commercially       **/
/**     Please, notify me, if you make any changes to this file            **/
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef MSDOS
#include <sys/stat.h>
#endif
#include "Bitmap.h"

typedef unsigned char byte;

#define FILE_NOT_FOUND          -3
#define READ_ERROR              -4
#define INVALID_FILE_LENGTH     -5
#define OUT_OF_MEMORY           -6

/****************************************************************************/
/* Read 8x8 character and put it in the bitmap buffer                       */
/****************************************************************************/
static void convert_cell (byte *pattern,byte *colour,byte *bmp,int width)
{
 int i,fg,bg,k;
 for (i=0;i<8;++i,pattern++,colour++,bmp+=width/2)
 {
  k=pattern[0];
  bg=colour[0]&0x0f;
  fg=(colour[0]>>4)&0x0f;
  if (k&0x80) bmp[0]=fg<<4; else bmp[0]=bg<<4;
  if (k&0x40) bmp[0]|=fg; else bmp[0]|=bg;
  if (k&0x20) bmp[1]=fg<<4; else bmp[1]=bg<<4;
  if (k&0x10) bmp[1]|=fg; else bmp[1]|=bg;
  if (k&0x08) bmp[2]=fg<<4; else bmp[2]=bg<<4;
  if (k&0x04) bmp[2]|=fg; else bmp[2]|=bg;
  if (k&0x02) bmp[3]=fg<<4; else bmp[3]=bg<<4;
  if (k&0x01) bmp[3]|=fg; else bmp[3]|=bg;
 }
}

/****************************************************************************/
/* Read a set 8x8 character and put them in the bitmap buffer. Skip first   */
/* two characters of every row, as these do not contain image data.         */
/****************************************************************************/
static void convert_cells (byte *pattern,byte *colour,byte *bmp,int width)
{
 int x,y;
 for (y=0;y<20;++y)
  for (x=2;x<32;++x)
   convert_cell (pattern+x*8+y*256,colour+x*8+y*256,
                 bmp+(x-2)*4+y*width*4,width);
}

/****************************************************************************/
/* Convert given file                                                       */
/****************************************************************************/
static int pp2bmp (char *filename,byte *palette)
{
 FILE *f;
 byte out[256],*p,*q,*pp,*bmp,*pattern,*colour;
 int filelen,width,height,i;
 /* If output file is given, use it. Else, use the input file with extension
    ".bmp" */
 p=strchr(filename,',');
 if (p)
 {
  strcpy (out,p+1);
  *p='\0';
 }
 else
 {
#ifdef MSDOS
/* If running on an MS-DOS machine, convert all backslashes to forward ones */
  _fixpath (filename,out);
#else
  strcpy (out,filename);
#endif
  p=out;
  q=strchr(p,'/');
  while (q)                             /* get last '/'                     */
  {
   p=++q;
   q=strchr(q,'/');
  };
  q=NULL;
  while ((p=strchr(p,'.'))!=NULL)       /* get last '.'                     */
  {
   q=p;
   ++p;
  }
  if (q) *q='\0';                       /* remove extension                 */
  strcat (out,".bmp");                  /* add ".bmp"                       */
 }
 printf ("Converting %s to %s... ",filename,out);
 fflush (stdout);
 f=fopen (filename,"rb");               /* open file                        */
 if (!f)
 {
  puts ("FAILED: File not found");
  return FILE_NOT_FOUND;
 }
 fseek (f,0,SEEK_END);                  /* check file length                */
 filelen=ftell (f);
 rewind (f);
 if (!filelen || (filelen%10240))
 {
  fclose (f);
  puts ("FAILED: Filelength is invalid");
  return INVALID_FILE_LENGTH;
 }
 filelen/=10240;
 pp=malloc ((filelen+1)*10240);         /* allocate buffer                  */
 if (!pp)
 {
  fclose (f);
  puts ("FAILED: Out of memory");
  return OUT_OF_MEMORY;
 }
 memset (pp,0,(filelen+1)*10240);       /* load the PowerPaint file         */
 if (fread(pp,1,filelen*10240,f)!=filelen*10240)
 {
  fclose (f);
  free (pp);
  puts ("FAILED: Read error");
  return READ_ERROR;
 }
 fclose (f);
 width=(filelen>1)? 480:240;            /* calculate image height and width */
 height=((filelen/2)+(filelen&1))*160;
 if (filelen==1)                        /* get pattern and colour table     */
 { pattern=pp; colour=pp+5120; }        /* offsets. note these differ       */
 else                                   /* between 10k binaries and 40/80k  */
 { colour=pp; pattern=pp+5120; }        /* workspace images                 */
 bmp=malloc (width*height/2);           /* allocate bitmap buffer           */
 if (!bmp)
 {
  free (pp);
  puts ("FAILED: Out of memory");
  return OUT_OF_MEMORY;
 }
 memset (bmp,0,width*height/2);
 for (i=0;i<filelen;++i)                /* convert all parts of the image   */
  convert_cells (pattern+i*10240,colour+i*10240,
                 bmp+(i&1)*120+(i/2)*240*160,width);
                                        /* now, write the bitmap file       */
 i=WriteBitmap (out,4,16,width,width,height,bmp,palette);
 if (i<0)
 {
  printf ("FAILED: ");
  switch (i)
  {
   case -1: puts ("Cannot open output file");
            break;
   default: puts ("Write error");
            break;
  }
 }
 else puts ("OK");
 free (bmp);
 free (pp);
 return i;
}

/****************************************************************************/
/* Check program arguments and convert all given files                      */
/****************************************************************************/
int main (int argc,char *argv[])
{
 char *Options[]= { "palette","bgcolour",NULL };
 int i,j,n;
 int misparm=0;
 static byte palettes[][16*3] =
 {
  /* Coleco ADAM's TMS9918 palette */
  {
     0,  0,  0,    0,  0,  0,   71,183, 59,  124,207,111,
    93, 78,255,  128,114,255,  182, 98, 71,   93,200,237,
   215,107, 72,  251,143,108,  195,205, 65,  211,218,118,
    62,159, 47,  182,100,199,  204,204,204,  255,255,255
  },
  /* Default V9938 palette */
  {
     0,  0,  0,    0,  0,  0,   32,192, 32,   96,224, 96,
    32, 32,224,   64, 96,224,  160, 32, 32,   64,192,224,
   224, 32, 32,  224, 96, 96,  192,192, 32,  192,192,128,
    32,128, 32,  192, 64,160,  160,160,160,  224,224,224
  },
  /* TMS9918 Black & White palette */
  {
     0,  0,  0,    0,  0,  0,  136,136,136,  172,172,172,
   102,102,102,  134,134,134,  120,120,120,  172,172,172,
   136,136,136,  172,172,172,  187,187,187,  205,205,205,
   118,118,118,  135,135,135,  204,204,204,  255,255,255
  },
  /* V9938 Black & White palette */
  {
     0,  0,  0,    0,  0,  0,  144,144,144,  195,195,195,
    60, 60, 60,  115,115,115,   80, 80, 80,  178,178,178,
   102,102,102,  153,153,153,  198,198,198,  211,211,211,
   100,100,100,  129,129,129,  182,182,182,  255,255,255
  }
 };
 int palnum=0;
 int bgcolour=1;
 printf ("pp2bmp: PowerPaint to Windows bitmap converter\n"
         "Copyright (C) Marcel de Kogel 1997\n"
         "This program supports both 10k binaries and "
         "40 and 80k workspace images\n");
 for (n=1,i=0;n<argc;n++)
 {
  if(*argv[n]!='-') ++i;
  else
  {    
   for(j=0;Options[j];j++)
    if(!strcmp(argv[n]+1,Options[j])) break;
   switch(j)
   {
    case 0:  n++;
             if (n<argc) palnum=atoi(argv[n]);
             else misparm=1;
             break;
    case 1:  n++;
             if (n<argc) bgcolour=atoi(argv[n]);
             else misparm=1;
             break;
    default: misparm=1;
             break;
   }
  }
 }
 if (palnum>3 || palnum<0) palnum=0;
 if (bgcolour<1 || bgcolour>15) bgcolour=1;
 if (misparm || !i)
 {
  printf ("Usage: pp2bmp [options] <file[,dest]> [file[,dest]...]\n"
          "Available options are:\n"
          "-palette <value>  - Select palette [0]\n"
          "                    0 - original\n"
          "                    1 - V9938\n"
          "                    2 - original/black and white\n"
          "                    3 - V9938/black and white\n"
          "-bgcolour <value> - Select background colour [1]\n");
  return 1;
 }
 palettes[palnum][0]=palettes[palnum][bgcolour*3+0];
 palettes[palnum][1]=palettes[palnum][bgcolour*3+1];
 palettes[palnum][2]=palettes[palnum][bgcolour*3+2];
 for (n=1;n<argc;n++)
 {
  if(*argv[n]!='-') pp2bmp(argv[n],palettes[palnum]);
  else
  {    
   for(j=0;Options[j];j++)
    if(!strcmp(argv[n]+1,Options[j])) break;
   switch(j)
   {
    case 0: case 1:
     n++;
    default:
     break;
   }
  }
 }
 return 0;
}
