/* The ADAM Connection
   The first IBM program that really simulates ADAM's EOS operating system!
   Can:
        * Delete EOS files from ADAM disks
        * Rename EOS files
        * Transfer directly from EOS to MS-DOS disks
        * Transfer directly from MS-DOS to EOS disks
   Compiles with Borland's Turbo C for DOS (which is available from Borland's antique
   software section of their website)

   (C) 1989 Dave White
   This software is hereby released into the public domain, effective February 25, 2009.
   It may be used for any purpose and redistributed in source or executable form.
*/

#include<stdlib.h>
#include<conio.h>
#include<stdio.h>
#include<bios.h>
#include<dos.h>
#include<fcntl.h>

#define AEOF -1                 /* Error codes for READFILE/WRITEFILE */
#define AERR -2

#define BUFSIZE 50              /* copy buffer size in K */

#define NOTFILE 1               /* attribute names */
#define EXEC 2
#define DELETED 4
#define SYSTEM 8
#define USER 16
#define READ 32
#define WRITE 64
#define ALL 128                 /* SW2 protection format (which is wrong) */

#define TRUE 1
#define FALSE 0

#define err(op) if (!(op)) return(FALSE)
        /* for proper error handling of non-transfer file routines */
#define werr(op) if (!(op)) { fclose(dosfile);\
                              cclosefcb(&fcb);\
                              return(FALSE); }
        /* for proper error handling of "write-to-EOS" routine */
#define rerr(op) if (!(op)) { cfclose(dosfile);\
                              return(FALSE); }
        /* for proper error handling of "read-from-EOS" routine */

int onedrive = FALSE;           /* one-drive system: do load/save differently */
int useallocd = FALSE;
struct text_info inforec;       /* for checking text mode */

struct eosfcb {         /* EOS file control block structure */
 char name[12];
 unsigned char attribute;
 unsigned int sblock;   /* start block #- really stored as 4 bytes */
 unsigned int sig;      /* two extra bytes for block #- only used for sig */
 unsigned int allocd;   /* blocks allocated */
 unsigned int used;     /* blocks used */
 unsigned int lcount;   /* last count */
 char date1;
 char date2;
 char date3;
                        /* NOT FOUND IN FILE DIRECTORY: */

 unsigned int offset;   /* offset of this FCB in file directory */
 unsigned char drive;   /* MS-DOS drive of this file */
 unsigned int dblock;   /* directory block number of this FCB */
 unsigned int rblock;   /* block currently being read from file */
 unsigned int roffset;  /* offset in given block */
 unsigned char dirsize; /* directory size */
 int fstat;             /* EOF or error */
 int bsize;             /* size of this block */
};

unsigned char order[4][2] = { {0,5},    /* ADAM disk block/sector conversion */
                              {2,7},
                              {4,1},
                              {6,3} };

byte dirbuff[1024];     /* block buffer for directory (like $D400) */
byte filebuff[1024];    /* block buffer for files (like $D800) */
byte copybuff[BUFSIZE][1024];   /* disk copy buffer. */
byte *currloc;          /* for buffer access */
unsigned int byteswritten;      /* for buffer access */
byte table[32];         /* formatting table */

unsigned int dboffs, dbseg;   /* location of disk base table */

int getkey()                    /* gets key, checks for CTRL-C */
{
 int x;
 x = bioskey(0);
 if (((char)x)==3) exit(0);             /* CTRL-C? Stop! */
 if (((char)x)==27) exit(0);            /* ESCAPE? Stop! */
 return(x);
}

int cwritebyte(byte x,struct eosfcb *fcb)
{
 if (!onedrive) return(writebyte(x,fcb));       /* two-drive system */
 if (byteswritten >= (BUFSIZE*1024)) {
  cputs("Not enough memory to load file!\n");
  return(FALSE);
 }
 byteswritten++;                /* one-drive system */
 *currloc = x;
 currloc++;
 return(TRUE);
}

int cclosefcb(struct eosfcb *fcb)       /* checks for one-drive system */
{
 if (onedrive) return(TRUE);            /* one-drive- don't close */
 else return(closefcb(fcb));
}

int cfclose(FILE *dosfile)              /* checks for one-drive system */
{
 if (onedrive) return(0);               /* don't do anything */
 else return(fclose(dosfile));
}

int cputc(int ch, FILE *dosfile)        /* checks for one-drive system */
{
 if (onedrive) {
  if (byteswritten >= (BUFSIZE*1024)) {
   cputs("Not enough memory to load file!\n");
   return(EOF);                         /* stream error */
  }
  byteswritten++;
  *currloc = (byte) ch;                 /* save byte in buffer */
  currloc++;
  return(0);                            /* success */
 }      /* end one-drive */
 else return(putc(ch,dosfile));
}

newline()               /* sends newline to console */
{
 cputs("\n");
}

ctextbackground(int color)      /* check for mono adapter first */
{
 if (inforec.currmode == MONO) return;  /* mono */
 textbackground(color);
}

ctextcolor(int color)           /* check for mono adapter first */
{
 if (inforec.currmode == MONO) return;  /* mono */
 textcolor(color);
}

byte *createformattable(int side, int track)    /* create BIOS table */
                                /* to format ADAM disk */
{
 int i,s;
 for (i=0,s=1; s<=8; s++) {     /* do 8 sectors */
  table[i]=track;       /* cylinder value */
  i++;
  table[i]=side;        /* head value */
  i++;
  table[i]=s;           /* record value */
  i++;
  table[i]=2;           /* size- 2 = 512 bytes/sector = 1/2 block */
  i++;
 }
 return(table);
}

int formattrack(int drive,int side, int track)  /* format ADAM track */
{
 int stat;

 pokeb(dbseg,dboffs,8);         /* set DBT "# sectors" to 8 for ADAM disks */
 createformattable(side,track);                 /* get format table */
 stat = biosdisk(5,drive,side,track,0,8,table); /* format track */
 pokeb(dbseg,dboffs,9);         /* reset DBT to MS-DOS standard */
 if (stat != 0) return(FALSE);
 return(TRUE);
}

byte *readblock(int drive, int block, byte *buff)       /* read ADAM block */
{
 int side,sblock,track,tblock, stat;

 pokeb(dbseg,dboffs,8);         /* set DBT "# sectors" to 8 for ADAM disks */
 side = block/160;
 sblock = block % 160;          /* % is C for "mod" */
 track = sblock/4;
 tblock = sblock % 4;
 stat = biosdisk(2,drive,side,track,order[tblock][0]+1,1,buff);  /* 1st sect */
 if (stat > 0) {
  pokeb(dbseg,dboffs,9);        /* restore DBT */
  biosdisk(0,drive,side,track,1,1,buff);
  return(NULL); /* NULL = disk access error */
 }
 biosdisk(2,drive,side,track,order[tblock][1]+1,1,buff+512); /* 2nd */
 if (stat > 0) {
  pokeb(dbseg,dboffs,9);        /* restore DBT */
  biosdisk(0,drive,side,track,1,1,buff);
  return(NULL); /* NULL = disk access error */
 }
 pokeb(dbseg,dboffs,9); /* restore DBT to MS-DOS format */
 return(buff);
}

byte *writeblock(int drive, int block, byte *buff) /* write ADAM block */
{
 int side,sblock,track,tblock, stat;

 pokeb(dbseg,dboffs,8);         /* set DCB "# sectors" to 8 for ADAM disks */
 side = block/160;
 sblock = block % 160;          /* % is C for "mod" */
 track = sblock/4;
 tblock = sblock % 4;
 stat = biosdisk(3,drive,side,track,order[tblock][0]+1,1,buff);  /* 1st sect */
 if (stat > 0) {
  pokeb(dbseg,dboffs,9);        /* restore DCB */
  biosdisk(0,drive,side,track,1,1,buff);
  return(NULL); /* NULL = disk access error */
 }
 stat = biosdisk(3,drive,side,track,order[tblock][1]+1,1,buff+512); /* 2nd */
 if (stat > 0) {
  pokeb(dbseg,dboffs,9);        /* restore DCB */
  biosdisk(0,drive,side,track,1,1,buff);
  return(NULL); /* NULL = disk access error */
 }
 pokeb(dbseg,dboffs,9); /* restore DCB to MS-DOS format */
 return(buff);
}

putvname(struct eosfcb *fcb)    /* displays volume name */
{
 char *i;

 i = fcb->name;
 while (*i != 3) { putch(*i); i++; }    /* print name */
}

putfname(struct eosfcb *fcb)    /* displays filename */
{
 char *i;

 i = fcb->name;
 if (*i==3) return;             /* null name */
 while (*(i+1) != 3) { putch(*i); i++; }        /* print name part */
 cprintf("(%c) ",*i);                           /* print type */
}

char *getfname(char *s)                 /* get filename */
{
 char buff[20]; int i,j;                /* temporary buffer, loop control */

 fflush(stdin); gets(buff);
 j = 0;
 for (i = 0; buff[i] != 0; i++)  {
  if (buff[i]=='(' || buff[i]==')') continue;   /* ignore parentheses */
  s[j]=buff[i]; j++;            /* j controls position in s */
 }
 s[j] = 0;      /* put end-text on s */
 return(s);
}

int checkdir(struct eosfcb *volume)     /* returns TRUE if disk is EOS disk, otherwise
                           FALSE. */
{
 if (volume->sblock != 0x0AA55) return(FALSE);  /* signature error */
 return(TRUE);                          /* an EOS disk! */
}

int firstfcb(int drive,struct eosfcb *fcb)    /* returns volume record */
{
 byte *buff;

 buff = readblock(drive,1,dirbuff);     /* get first directory block */
 if (buff==NULL) return(FALSE);         /* error */
 memcpy(fcb,buff,26);                   /* copy FCB data to buffer */
 fcb->drive = drive;
 fcb->offset = 0;
 fcb->dblock = 1;
 fcb->dirsize = (fcb->attribute) & 0x7F;  /* get directory size */
 return(TRUE);
}

int nextfcb(int drive,struct eosfcb *fcb)   /* gets fcb after current one */
{
 byte *buff;

 if ((fcb->offset + 52) > 1023) {               /* if next is in new block */
  (fcb->dblock)++;
  if (fcb->dblock > fcb->dirsize) return(FALSE);  /* went too far */
  fcb->offset = 0;
  buff = readblock(drive,1,dirbuff);
  if (buff==NULL) return(FALSE);                /* error loading block */
 }
 else                                           /* if it's in the same block */
  fcb->offset += 26;
 memcpy(fcb,dirbuff+(fcb->offset),26);                  /* get FCB */
 return(TRUE);
}

int comparefcb(struct eosfcb *fcb1, char *name)
                                /* TRUE = same, FALSE = different */
                                /* NAME is C-type string; fcb1 is
                                   EOS-type FCB string */
{
 char *i,*j;

 for (i = fcb1->name, j = name;
      *i != 3 && *j != 0 && *i == *j;   /* equal or end of string */
      i++, j++
     );
 if (*j == 0 && *i == 3) return(TRUE);  /* true if both are at end */
 return(FALSE);
}

int writefcb(struct eosfcb *fcb) /* writes an FCB back into the directory */
{
 err(readblock(fcb->drive,fcb->dblock,dirbuff));   /* get old dir block */
 memcpy(dirbuff+(fcb->offset),fcb,26);             /* transfer data */
 err(writeblock(fcb->drive,fcb->dblock,dirbuff));  /* write new block */
 return(TRUE);
}

int findfcb(int drive, struct eosfcb *fcb, char *name)  /* finds name */
                                /* can be anything but a deleted file */
{

 err(firstfcb(drive, fcb));                     /* FirstFCB w/errcheck */
 err(checkdir(fcb));                    /* Check volume record */
 fcb->name[0] = 3;      /* clear name so volume won't be matched */
 while (!comparefcb(fcb,"BLOCKS LEFT")) {
  if (((fcb->attribute) & DELETED) == 0) if (comparefcb(fcb,name)) break;
  err(nextfcb(drive,fcb));                      /* move to next record */
 }
 if (!strcmp(name,"BLOCKS LEFT")) return(TRUE);
 if (comparefcb(fcb,name)) return(TRUE); /* found */
 return(FALSE); /* not found */
}

int setfcb(struct eosfcb *fcb, char *fname, unsigned int ballocd,
  unsigned int bused)
{
 char *i;

 memcpy(fcb,fname,12);  /* copy name */
 for (i = fcb->name; *i != 0; i++); *i = 3;     /* change EOT */
 fcb->attribute = 16;   /* user file */
 if (ballocd != 0) fcb->allocd = ballocd;  /* 0=don't change */
 fcb->used = bused;
 fcb->lcount = 0;
 err(writefcb(fcb));
 return(TRUE);
}

int makefcb(int drive, char *name, long size)
                /* create new directory entry */
{
 struct eosfcb fcb, blocksleft;
 unsigned int blocks;   /* blocks required for file of given size */

 blocks = (unsigned int) ((size)/((unsigned long)1024)+((unsigned long)1));
 if (blocks == 0) blocks = 2048;  /* for unknown size file */
 err(firstfcb(drive, &fcb));                    /* FirstFCB w/errcheck */
 err(nextfcb(drive,&fcb));                      /* Get first file record */
 while (!comparefcb(&fcb,"BLOCKS LEFT")) {
  if (((fcb.attribute & DELETED)!=0) && (fcb.allocd >= blocks)) break;
                /* found deleted file with enough space */
  err(nextfcb(drive,&fcb));                     /* move to next record */
 }
 if (blocks == 2048) blocks = fcb.allocd;  /* allocate unknown size file all available space */
 if (fcb.allocd < blocks) {
  cputs("No more space on EOS disk!\n");
  return(FALSE);
 }
 if (comparefcb(&fcb,"BLOCKS LEFT")) {  /* No deleted file- add record */
  if ((fcb.offset+52 > 1023) && ((fcb.dblock+1) > fcb.dirsize)) {
   cputs("No more space on EOS directory!\n");
   return(FALSE);
  }
  memcpy(&blocksleft,&fcb,sizeof(struct eosfcb));   /* copy end FCB */
  err(setfcb(&fcb,name,blocks,1));   /* set it to new values, write to disk */
  err(nextfcb(drive,&fcb));             /* go to next one */
  memcpy(&fcb,&blocksleft,26);  /* copy end data to new record */
  fcb.sblock += blocks;         /* move start block up appropriately */
  fcb.allocd -= blocks;         /* adjust blocks free */
  writefcb(&fcb);               /* write it to disk */
 }
 else   /* Deleted file, set to new file values */
  err(setfcb(&fcb,name,0,1));   /* 0 as alloc'd means don't change. */
 return(TRUE);          /* Done allocating new space. */
}


int openfcb(int drive, struct eosfcb *fcb, char *name)    /* opens file */
{
 err(findfcb(drive,fcb,name));  /* find name */
 fcb->rblock = fcb->sblock;     /* set up file-read fields */
 fcb->roffset = 0;
 fcb->fstat = 0;                        /* File status */
 fcb->bsize = 1024;             /* # bytes in this block */
 if (fcb->used == 1) fcb->bsize = fcb->lcount;  /* less for 1-block file */
 if (fcb->used <= 1 && fcb->lcount == 0) fcb->fstat = AEOF; /* null file */
 if (readblock(drive,fcb->sblock,filebuff)==NULL) return(FALSE);
                        /* read first block of file */
 return(TRUE);
}

int eosrename(int drive, char *name1, char *name2)  /* renames file */
{
 struct eosfcb fcb;

 err(findfcb(drive,&fcb,name1));        /* find old name */
 memcpy(&fcb,name2,12); /* copy new name */
 (fcb.name)[11] = 0;    /* ETX in case name is too long */
 (fcb.name)[strlen(fcb.name)] = 3;      /* convert ETX from 0 to 3 */
 err(writefcb(&fcb));   /* write to disk */
 return(TRUE);
}

int delete(int drive, char *fname)      /* deletes file */
{
 struct eosfcb fcb;

 if (!findfcb(drive,&fcb,fname)) return(TRUE);  /* always returns TRUE */
 fcb.attribute |= DELETED;      /* delete the file */
 err(writefcb(&fcb));
 return(TRUE);
}

int backup(int drive, char *fname)      /* makes file a backup */
{
 char fname2[15];       /* holds 'backup' file name */

 if (fname[0]==0) return(FALSE);  /* no name! */
 strncpy(fname2,fname,12);
 fname2[strlen(fname2)-1] += 32;   /* lowerfy the type letter */
 err(delete(drive,fname2));             /* delete old backup file */
 err(eosrename(drive,fname,fname2));    /* rename main file to backup */
 return(TRUE);
}

int wopenfcb(int drive, struct eosfcb *fcb, char *name, long size)
                        /* opens for writing */
{
 if (findfcb(drive,fcb,name))
  err(backup(drive,name));      /* already there-make backup copy */
 err(makefcb(drive,name,size));
 err(openfcb(drive,fcb,name));
 fcb->bsize = 1024;             /* is set to 0 by open-to-read routine- must
                                   be 1024 for writing */
 return(TRUE);
}

int readbyte(struct eosfcb *fcb)        /* returns 1 byte read from file */
                                        /* returns AEOF on end-file */
                                        /* returns AERR on error */
{
 byte x;
 int relblock;  /* relative block # */

 if (fcb->fstat != 0) return(fcb->fstat);  /* EOF or error */
 x = *(filebuff + fcb->roffset);
 (fcb->roffset)++;              /* next byte */
 if (fcb->roffset >= fcb->bsize) {              /* end of block? */
  (fcb->rblock)++;
  fcb->roffset = 0;
  relblock = (fcb->rblock)-(fcb->sblock)+1;
  if (useallocd)
    {
    /* user can set EXPAND environment variable to copy all the */
    /* space allocated to the file (even that which is not used) */
    if (relblock == fcb->allocd) fcb->bsize = 1024;
    if (relblock > fcb->allocd) fcb->fstat = AEOF;
    }
  else
    {
    if (relblock == fcb->used) fcb->bsize = fcb->lcount;   /* last block rtn */
    if (relblock > fcb->used) fcb->fstat = AEOF;          /* EOF rtn */
    }
  if (fcb->fstat != AEOF)               /* read block if not eof */
   if (readblock(fcb->drive,fcb->rblock,filebuff) == NULL)
    fcb->fstat = AERR;
 }              /* end "end of block" routine */
 return( (int) x);
}

selectdrive(char *msg, int *drive)      /* selects drive, stores at location */
{
 char choice;

 if (onedrive) {        /* only one possible drive */
  *drive = 0;
  return;
 }
 cputs(msg);
 choice = 0;
 while (toupper(choice)<'A' || toupper(choice)>'B') {
  choice = (char) getkey(); fflush(stdin);
 }
 *drive = (toupper(choice))-'A';
 cputs("\n");
}

int copydisk(drive)             /* Copies 1 (one) ADAM disk. */
{
 int rblock, wblock;            /* READ block and WRITE block */
                                /* For block-move-as-copying commands */
 int block, volsize, i;
 int drive2;
 int buffsize=BUFSIZE, buffstart, blockswritten;

 ctextbackground(GREEN);
 ctextcolor(WHITE);
 clrscr();
 cputs("COPY AN ADAM DISK\n");
 ctextcolor(YELLOW);
 cputs("\n");
 selectdrive("Copy to which drive (A or B)?",&drive2);
 cprintf("Start block for reading data (0 for normal copy)?");
 fflush(stdin); scanf("%d",&rblock);
 cprintf("Start block for writing data (0 for normal copy)?");
 fflush(stdin); scanf("%d",&wblock);
 cprintf("Copy how many blocks (160 = single-sided, 320 = double-sided)?");
 fflush(stdin); scanf("%d",&volsize);
 if (drive != drive2) {         /* Message for two-drive system. */
  cprintf("Put the disk to be copied into Drive %c;\n",(char)(drive+'A'));
  cprintf("Put a new disk into Drive %c.\n",(char)(drive2+'A'));
 } else
  cprintf("Put the disk to be copied into Drive %c;\n",(char)(drive+'A'));
 cputs("Press RETURN to begin copying...\n");
 getkey(); fflush(stdin);
 currloc = (byte *) copybuff;                   /* locations to read/write */
 blockswritten = 0;                     /* # K in buffer */
 buffstart = wblock;                    /* first block # in buffer */
 ctextcolor(WHITE);
 for (block = rblock; !(block >= volsize && blockswritten == 0);
                wblock++, block++) {
  if (blockswritten == buffsize || block >= volsize) {
                                /* buffer filled-write data */
   if (drive == drive2) {       /* one-drive routine */
    cprintf("Put the blank disk into Drive %c"
            " and press any key...\n",drive2+'A');
    getkey(); fflush(stdin);
   }
   currloc = (byte *) copybuff;                         /* start at beginning */
   for (i = buffstart; i < wblock; i++) {
    gotoxy(1,wherey());
    ctextbackground(BROWN);             /* BROWN = writing; RED = reading */
    cprintf("Writing block %d...",i);
    if (NULL==writeblock(drive2,i,currloc)) return(FALSE);
    currloc += 1024;
   }
   blockswritten = 0;
   currloc = (byte *) copybuff;
   buffstart = wblock;
   if (drive == drive2) {               /* one-drive routine */
    cprintf("Put the disk to be copied from"
            " into Drive %c and press any key...\n",drive+'A');
    getkey(); fflush(stdin);
   }
  }                     /* end write routine */
  if (block >= volsize) break;  /* for last part of copy */
  gotoxy(1,wherey());           /* print status report */
  ctextbackground(RED);         /* RED = reading; BROWN = writing */
  cprintf("Reading block %d...",block);
  if (NULL==readblock(drive,block,currloc))
        return(FALSE);  /* read block */
  blockswritten++;
  currloc += 1024;
 }                              /* end main FOR loop */
 ctextbackground(GREEN);
 ctextcolor(YELLOW);
 return(TRUE);
}

int formatdisk(drive)   /* Formats 1 (one) ADAM disk. */
{
 int track, side, sides, dirsize, volsize;
 char volname[15];
 struct eosfcb fcb;

 ctextbackground(MAGENTA);
 clrscr();
 ctextcolor(WHITE);
 cputs("FORMAT ADAM DISK\n");
 cputs("\n");
 ctextcolor(YELLOW);
 cprintf("Enter a volume name: ");
 fflush(stdin); gets(volname);
 cprintf("Enter the media size (160 = single-sided, 320 = double-sided):");
 fflush(stdin); scanf("%d",&volsize);
 cprintf("Enter the directory size (1-3):");
 fflush(stdin); scanf("%d",&dirsize);
 newline();
 if (dirsize > 3) dirsize = 3;
 if (drive == 0) cputs("Put the disk to be formatted in Drive A...\n");
 else cputs("Put the disk to be formatted iin Drive B...\n");
 cputs("Press RETURN to begin...\n");
 getkey(); fflush(stdin);
 if (volsize > 160) sides = 2; else sides = 1;
 ctextcolor(WHITE);
 ctextbackground(RED);
 for (side = 0; side < sides; side++)
  for (track = 0; track < 40; track++) {
   gotoxy(1,wherey());          /* put at left of screen */
   cprintf("Formatting track %d...\0x0D",track);
   err(formattrack(drive,side,track));
  }
 ctextbackground(MAGENTA);
 newline();
 cputs("Initializing disk directory...\n");
 err(firstfcb(drive,&fcb));
 strcpy(fcb.name,volname);      /* copy volume name */
 fcb.name[strlen(volname)] = 3; /* use proper ETX for ADAM system */
 fcb.sblock = 0x0AA55;          /* signature */
 fcb.sig = 0xFF00;
 fcb.attribute = 128 | (dirsize);
 fcb.allocd = volsize;          /* volume size (4 bytes) */
 fcb.used = 0;
 fcb.date1 = fcb.date2 = fcb.date3 = 0;
 err(writefcb(&fcb));           /* write volume record */
 err(nextfcb(drive,&fcb));
 strcpy(fcb.name,"BOOT\003");   /* BOOT record */
 fcb.sblock = 0;
 fcb.sig = 0;
 fcb.attribute = 0x088;
 fcb.allocd = fcb.used = 1;
 fcb.lcount = 3;
 fcb.date1 = fcb.date2 = fcb.date3 = 0;
 err(writefcb(&fcb));
 err(nextfcb(drive,&fcb));
 strcpy(fcb.name,"DIRECTORY\003");      /* DIRECTORY record */
 fcb.sblock = 1;
 fcb.sig = 0;
 fcb.attribute = 0x0C8;
 fcb.allocd = dirsize;
 fcb.used = dirsize;
 fcb.lcount = 1024;
 err(writefcb(&fcb));
 err(nextfcb(drive,&fcb));
 strcpy(fcb.name,"BLOCKS LEFT\003");
 fcb.sblock = dirsize+1;
 fcb.sig = 0;
 fcb.attribute = 0x01;
 fcb.allocd = volsize-dirsize-1;        /* blocks left */
 fcb.used = 0;
 fcb.lcount = 0;
 err(writefcb(&fcb));
 err(nextfcb(drive,&fcb));
 return(TRUE);                          /* done */
}


int closefcb(struct eosfcb *fcb)        /* closes file after writing */
{
 unsigned int relblock;         /* relative block number */

 relblock = fcb->rblock - fcb->sblock + 1;
 if (relblock > 1 && fcb->roffset == 0) {       /* eliminate empty blocks */
  relblock--;
  fcb->roffset = 1024;
 }
 err(writeblock(fcb->drive,fcb->rblock,filebuff));  /* write last block */
 fcb->used = relblock;          /* set # blocks used */
 fcb->lcount = fcb->roffset;            /* set last count */
 err(writefcb(fcb));            /* write to disk */
 return(TRUE);
}

int writebyte(byte x, struct eosfcb *fcb)       /* writes 1 byte to file */
                /* TRUE = success, FALSE = failure */
{
 int relblock;  /* relative block # */

 if (fcb->fstat == AERR) {
  closefcb(fcb);
  cputs("Write past end!\n");
  return(FALSE);  /* error */
 }
 filebuff[fcb->roffset] = x;
 fcb->roffset++;                /* next byte */
 if (fcb->roffset >= fcb->bsize) {              /* end of block? */
  putch((int) '.');             /* one . for each block written */
  if (writeblock(fcb->drive,fcb->rblock,filebuff) == NULL) {
   closefcb(fcb);
   return(FALSE);   /* error writing block */
  }
  (fcb->rblock)++;
  fcb->roffset = 0;
  relblock = (fcb->rblock)-(fcb->sblock)+1;
  if (relblock > fcb->allocd) fcb->fstat = AERR; /* Write past end? */
 }              /* end "end of block" routine */
 return(TRUE);
}


int save(int drive, char *name, char *dospath)
                /* Transfers file from MS-DOS to EOS. */
{
 struct eosfcb fcb;
 int doshandle; /* File handles required for getting file size */
 FILE *dosfile;
 int stat; int sw = FALSE; int smart = FALSE; int ascii = FALSE; char c;
 int merge = FALSE;     /* TRUE = merge lines into paragraphs */
 long fsize, i;
 char lastchar;         /* holds last character read- a space? omit CRLF */
 byte *wpos;            /* for one-drive buffer write */

 if (onedrive) {
  cputs("Put in MS-DOS disk and press any key...\n");
  getkey(); fflush(stdin);
 }
 byteswritten = 0; currloc = (byte *) copybuff;
                                /* set up for one-drive copy */
                                        /* (just in case) */
 lastchar = 0;
 ascii = sw = FALSE;
 clrscr();
 ctextcolor(WHITE);
 puts("What kind of IBM file is this?");
 ctextbackground(RED);
 cputs("1. A text file - don't reformat                     \n");
 cputs("2. A text file to be reformatted for SpeedyWrite    \n");
 cputs("3. A text file to be reformatted for SmartWRITER    \n");
 cputs("4. Not a text file                                  \n");
 ctextbackground(BLUE);
 cputs("Select 1 to 4:");
 c = 0;
 while (c < '1' || c > '4') {
  c = (char)getkey();
  fflush(stdin);
 }
 switch (c) {
  case '1': ascii = TRUE; break;
  case '2': ascii = TRUE; sw = TRUE; break;
  case '3': ascii = TRUE; smart = TRUE; break;
 }
 if (smart || sw) {                     /* wp conversion factor */
  cputs(
    "\nPut RETURN symbols only at end of paragraphs (Press Y if unsure)? ");
  merge = (toupper(bioskey(0))=='Y'); fflush(stdin);
 }
 doshandle = open(dospath,O_RDONLY | O_BINARY );
 fsize = (long) filelength(doshandle);          /* get size */
 close(doshandle);
 if (fsize == -1) return(FALSE);        /* error getting size */
 while (fsize < 0) fsize += 65536;      /* math correction ($1500 and the
                                           blasted computer can't add! */
 dosfile = fopen(dospath,"rb");         /* open binary DOS file for reading */
 if (dosfile == NULL) return(FALSE);

 if (!onedrive)                 /* open now only for two-drive systems */
  werr(wopenfcb(drive,&fcb,name,fsize));                /* open EOS file */
 cputs("Writing to EOS file... \n");
 for (i=1; i <= fsize; i++)     {               /* write fsize bytes */
  c = getc(dosfile);
  if (c == EOF) {
   cclosefcb(&fcb);
   fclose(dosfile);
   return(FALSE);
  }
  if (ascii && c==10) continue;  /* skip linefeeds for ASCII files */
  if ((sw || smart) && merge && c==13 && lastchar == 32)
            /* carriage return */
   continue;
  if (smart && c==12) {         /* convert IBM end-page to SmartWR command */
   werr(cwritebyte(12, &fcb));
   werr(cwritebyte(13, &fcb));
   lastchar = 13;
   continue;
  }
  if (sw && c==12) {              /* convert IBM end-page to SW command */
   werr(cwritebyte((byte)('E'+128), &fcb));  /* write CTRL-V+"E" */
   werr(cwritebyte(13,&fcb));               /* write RETURN */
   lastchar = 13;
   continue;
  }
  if (ascii && c==26) break;    /* ASCII end-of-file: stop */
  if (sw && c=='\\') {            /* SW2- special IBM code for CTRL-V */
   c = getc(dosfile);
   if (c == EOF) {
    cclosefcb(&fcb);
    fclose(dosfile);
    return(FALSE);
   }
   c += 128;                    /* convert to real CTRL-V */
  }
  werr(cwritebyte( (byte) c, &fcb));    /* transfer byte */
  lastchar = c & 127;                   /* last character = last ASCII char */
 }
 fclose(dosfile);
 err(cclosefcb(&fcb));          /* close EOS file */
 if (onedrive) {                        /* one drive- save buffer */
  cputs("Put in ADAM disk and press any key...\n");
  getkey(); fflush(stdin);
  werr(wopenfcb(drive,&fcb,name,fsize));        /* open ADAM file */
  for (wpos =(byte *) copybuff; wpos < currloc; wpos++) { /* write loop */
   werr(writebyte(*wpos, &fcb));        /* write byte */
  }             /* end write loop */
  err(closefcb(&fcb));  /* close ADAM file */
 }              /* end one-drive routine */
 fclose(dosfile);
 return(TRUE);          /* success! */
}

int writeln(FILE *dosfile, char *aline, int i)  /* Writes an ASCII line. */
{
 int stat, k; char *j;

 i--;
 while(i>0 && aline[i]==32) i--;        /* find last non-space */
 i++; aline[i]=0;               /* place end-of-string char */
 if (strchr(aline,13)!=NULL) aline[strchr(aline,13)]=0;
                        /* replace a RETURN-symbol with a 0 */
 if (onedrive) {                        /* one-drive system */
  for (j=aline; (*j)!=0; j++) {
   stat = cputc(*j,dosfile);
   if (stat==EOF) return(FALSE);
  }
  stat = cputc(13,dosfile); if (stat==EOF) return(FALSE);
  stat = cputc(10,dosfile); if (stat==EOF) return(FALSE);
 }
 else {                                 /* two-drive system */
  stat = fprintf(dosfile,"%s\x0D\x0A",aline);
  if (stat == EOF) {
   fclose(dosfile);
   return(FALSE);
  }
 }
 return(TRUE);
}

#define lerr(op) if (op==-1) { \
        cputs("Error in accessing MS-DOS file!\n");\
        fclose(dosfile); return(FALSE); }


int load(int drive, char *name, char *dospath)
                /* Transfers file from EOS to MS-DOS. */
{
 struct eosfcb fcb;
 FILE *dosfile;
 int stat; char c,c1,c2,c3,width; int i;
 byte aline[161];               /* line array for reading SmartWRITER files */
 byte *wpos;                    /* for writing buffer with one-drive sys */
 int ascii, sw, fix;            /* ASCII Text & SpeedyWrite flags */

 byteswritten = 0; currloc =(byte *) copybuff;
                 /* in case of one-drive system */
 fix = 0;
 if (onedrive) {
  cputs("Put in ADAM disk and press any key...\n");
  getkey(); fflush(stdin);
 }
 ascii = sw = FALSE;            /* default = binary file */
 if (!onedrive)
  dosfile = fopen(dospath,"wb");    /* open binary DOS file for writing */
 if (!onedrive && dosfile == NULL) return(FALSE);

 if (strlen(name)==0) return(FALSE);            /* null name */
 rerr(openfcb(drive,&fcb,name));                        /* open EOS file */
 if (toupper(name[strlen(name)-1])=='H')  {     /* H-type file? */
  c1 = readbyte(&fcb); lerr(c1);        /* get first 3 bytes */
  c2 = readbyte(&fcb); lerr(c2);
  c3 = readbyte(&fcb); lerr(c3);        /* c3 = id byte */
  if (c3==1) {                  /* SmartWRITER ID */
   cprintf("%s is a SmartWRITER file... Converting to ASCII text...\n",name);
   c2 = readbyte(&fcb); lerr(c2);  /* offset of 3 */
   c2 = readbyte(&fcb); lerr(c2);  /* offset of 4 */
   c2 = readbyte(&fcb); lerr(c2);  /* left margin */
   c3 = readbyte(&fcb); lerr(c3);  /* right margin */
   width = c3-c2+1;                                 /* line width */
   for (i=1; i<=252; i++) readbyte(&fcb);       /* skip header */
   i = 0;                                       /* column position */
   while (fcb.fstat == 0) {             /* file-reading loop */
    c = readbyte(&fcb); lerr(c);
    if (c==255) c=26;           /* \xFF is Sm.W's EOF */
    if ((c==19) || (c==20)) c='_';      /* underline */
    if (c==13 && i==0) {                /* 'solo entry' CR (as they say) */
     aline[0]=0;                                /* send blank line */
     err(writeln(dosfile,aline,i));
     continue;                                  /* skip rest of loop */
    }
    if (c==13) {                        /* end-of-paragraph CR */
     err(writeln(dosfile,aline,i));
     i = 0;
     continue;
    }
    aline[i]=c; i++;
    if (i>=width)  {                    /* end-of-line */
     err(writeln(dosfile,aline,i));     /* write the line */
     i = 0;
    }   /* end write line */
   }                            /* end main read loop */
   err(writeln(dosfile,aline,i));       /* write last line */
   cputc(26,dosfile);                   /* EOF */
   if (cfclose(dosfile)==EOF) return(FALSE);
   if (onedrive) {                      /* one drive- save buffer */
    cputs("Put in MS-DOS disk and press any key...\n");
    getkey(); fflush(stdin);
    dosfile = fopen(dospath,"wb");      /* open DOS file */
    if (dosfile==NULL) return(FALSE);   /* open error */
    for (wpos =(byte *) copybuff; wpos < currloc; wpos++) {  /* write loop */
     stat = putc((int) *wpos, dosfile); /* write byte */
     if (stat == EOF) {
      fclose(dosfile);
      return(FALSE);
     }
    }           /* end write loop */
    fclose(dosfile);
   }            /* end one-drive routine */
   return(TRUE);                        /* end read SmartWRITER file */
  } else {
   cputs("Copying binary file...\n");
   cputc(c1,dosfile);   /* Non-SmartWRITER H-file: write header bytes */
   cputc(c2,dosfile);
   stat = cputc(c3,dosfile);    /* only need error stat on last write */
   if (stat == EOF) {
    cfclose(dosfile);
    return(FALSE);
   }
  }             /* end binary file code */
 }                                      /* End H-file code */
 if (toupper(name[strlen(name)-1])!='H') {      /* Non-header file */
  ctextcolor(WHITE);
  clrscr();
  cputs("What kind of file is this?\n");
  ctextbackground(RED);                 /* file type menu */
  cputs("1. SmartBASIC program or data file       \n");
  cputs("2. SpeedyWrite or SpeedyWrite 2 text file\n");
  cputs("3. Other                                 \n");
  ctextbackground(BLUE);
  cputs("Select from 1 to 3:\n");
  c = 0;
  while (c<'1' || c>'3') {
   c = (char) getkey(); fflush(stdin); /* get choice */
  }
  switch (c) {
   case '1': ascii=TRUE; sw=FALSE; break;
   case '2': ascii=TRUE; sw=TRUE;
             cputs("\nPut spaces at beginning of paragraphs? ");
             fix = (toupper(bioskey(0))=='Y'); fflush(stdin); break;
   case '3': ascii=FALSE; break;        /* binary file */
  }
 }              /* end ASCII submenu */
 if (fix) cputc(32,dosfile);
 while (fcb.fstat == 0) {               /* read till AEOF or AERR */
  c = readbyte(&fcb); lerr(c);
  if (ascii && c==13)   {                       /* ASCII conversions */
    cputc(13,dosfile);
    stat = cputc(10,dosfile);
    if (fix) stat = cputc(32,dosfile);
                        /* for SW2, helps IBM WP's reformat text */
  }
  else if (sw && c>127) {               /* SpeedyWrite conversions */
    cputc('\\',dosfile);                /* CTRL-V -> backslash */
    stat = cputc(c-128,dosfile);
  }
  else stat = cputc(c,dosfile);         /* binary file- transfer byte */
  if (stat == EOF) {
   cfclose(dosfile);
   return(FALSE);
  }
 }
 if (ascii) cputc(26,dosfile);          /* EOF for ASCII files */
 cfclose(dosfile);                      /* close DOS file */
 if (onedrive) {                        /* one drive- save buffer */
  cputs("Put in MS-DOS disk and press any key...\n");
  getkey(); fflush(stdin);
  dosfile = fopen(dospath,"wb");        /* open DOS file */
  if (dosfile==NULL) return(FALSE);     /* open error */
  for (wpos =(byte *) copybuff; wpos < currloc; wpos++) { /* write loop */
   stat = putc((int) *wpos, dosfile);   /* write byte */
   if (stat == EOF) {
    fclose(dosfile);
    return(FALSE);
   }
  }             /* end write loop */
  cfclose(dosfile);
 }              /* end one-drive routine */
 if (fcb.fstat == AERR) return(FALSE);
 return(TRUE);
}



int filestat(int drive, char *name)     /* displays File Status as in SW2 */
{
 int stat;
 struct eosfcb fcb;
 char choice;           /* for attribute change */

 while(TRUE) {          /* for multiple status changes */
  ctextbackground(BLACK);
  clrscr();
  ctextcolor(LIGHTGREEN);
  cputs("STATUS OF FILE\n");
  ctextcolor(YELLOW);
  cprintf("Filename: %s\n",name);
  stat = findfcb(drive,&fcb,name);
  if (!stat) return(FALSE);

  cprintf("Attribute: %X\n",fcb.attribute);
  ctextbackground(GREEN);
  ctextcolor(WHITE);
  if ((fcb.attribute & NOTFILE) != 0)   cputs("NOT FILE              \n");
  if ((fcb.attribute & DELETED) != 0)   cputs("DELETED               \n");
  if ((fcb.attribute & READ) != 0)      cputs("READ PROTECTED        \n");
  if ((fcb.attribute & WRITE) != 0)     cputs("WRITE PROTECTED       \n");
  if ((fcb.attribute & ALL) != 0)       cputs("ALL (DELETE) PROTECTED\n");
  if ((fcb.attribute & EXEC) != 0)      cputs("EXEC PROTECTED        \n");
  if ((fcb.attribute & USER) != 0)      cputs("USER FILE             \n");
  if ((fcb.attribute & SYSTEM) != 0)    cputs("SYSTEM FILE           \n");
  ctextbackground(BLACK);
  ctextcolor(WHITE);
  cputs("\n");
  cprintf("Start block of file: %X hex, %d dec\n",fcb.sblock,fcb.sblock);
  if (fcb.used == 0) cprintf("File size (bytes): 0\n"); else
   cprintf("File size (bytes): %lu\n",(long)((fcb.used-1)*1024)+fcb.lcount);
  cprintf("Blocks allocated to file: %X hex, %d dec\n",fcb.allocd,fcb.allocd);
   cprintf("Blocks used: %X hex, %d dec\n",fcb.used,fcb.used);
  cprintf("Bytes in last block: %X hex, %d dec\n",fcb.lcount,fcb.lcount);
  cprintf("Date: %X %X %X\n",fcb.date1,fcb.date2,fcb.date3);
  ctextcolor(WHITE);
  cputs("Status Change Menu- Press Appropriate Number to Change Status\n");
  cputs("WARNING: Do NOT remove the disk until you have left STATUS!\n");
  ctextbackground(RED);
  cputs("1. END-OF-DIRECTORY FILE (CAUTION!)    6. READ PROTECT           \n");
  cputs("2. EXEC PROTECT                        7. WRITE PROTECT          \n");
  cputs("3. DELETED                             8. LOCK (ALL PROTECT      \n");
  cputs("4. SYSTEM FILE                             in SpeedyWrite)       \n");
  cputs("5. USER FILE                           9. Exit Status Command    \n");
  ctextbackground(BLACK);
  cputs("Select 1-9:\n");
  choice = 0;                                   /* get valid menu selection */
  while (choice < '1' | choice > '9') {
   choice = (char) getkey(); fflush(stdin);
  }
  if (choice == '9') return(TRUE);              /* Exit */
  fcb.attribute ^= (1 << (choice-'1')); /* compute mask & use OR to set bit */
  err(writefcb(&fcb));                  /* write new FCB */
 }                              /* end main while loop */
}


int catalog(int drive)  /* displays catalog of drive */
{
 struct eosfcb fcb;
 int row;

 err(firstfcb(drive,&fcb));     /* get start */
 ctextbackground(BROWN);
 ctextcolor(WHITE);
 clrscr();
 cputs("VOLUME NAME: ");
 putvname(&fcb); cprintf("\n");
 err(nextfcb(drive,&fcb));
 row = 0;               /* to make sure names don't scroll */
 while (!comparefcb(&fcb,"BLOCKS LEFT")) {      /* till end */
  putfname(&fcb);
  if ((fcb.attribute & DELETED) != 0) cprintf("(DELETED)");
  cputs("\n");
  err(nextfcb(drive,&fcb));
  row++;
  if (row == 23) {
   row = 0;
   cputs("PRESS ANY KEY FOR NEXT SCREEN...");
   getkey(); fflush(stdin);
   clrscr();
  }
 }
}

finddiskbase()          /* finds disk base table in any DOS */
{
 dboffs = peek(0,120)+4;        /* get data from interrupt 30 */
                                /* # sectors (only difference btwn. ADAM & IBM)
                                   is at offset of 4 */
 dbseg = peek(0,122);
 if (dbseg >= 0xF000) {
  cputs("You need a DOS version higher than 1.0 to run this program.\n");
  exit(1);
 }
}

main(int argc, char *argv[])                    /* main program file */
                        /* accepts DOS argument "1" for one-drive system */
{
 char choice;
 int drive = 0;         /* default = Drive A */
 char name[14];         /* EOS name */
 char newname[14];      /* New EOS Name for Rename command */
 char path[80];         /* DOS name */
 int stat;

 /* simmain(0); */  /* initialize the Z80 simulator */
 if ((argc > 1) && (!strcmp(argv[1],"1")))
  onedrive = TRUE;                      /* one-drive paramter */
 if (getenv("EXPAND") != NULL)
  useallocd = TRUE;  /* expanded ADAM file reading environment variable */
 gettextinfo(&inforec);         /* check if system has a mono adapter */
 while(TRUE) {
  finddiskbase();       /* find the disk base table */
  ctextbackground(BLUE);
  ctextcolor(CYAN);
  clrscr();
  cputs("Welcome to the ADAM Disk Manager\n");
  cputs("--------------------------------\n");
  cputs("Direct Access to ADAM EOS Disks from IBM MS-DOS!\n");
  cputs("Accepts either Single or Double-Sided Disks!\n");
  cputs("\n");
  cprintf("Current drive is %c:.\n",'A'+drive);
  cputs("\n");
  ctextbackground(RED);
  ctextcolor(WHITE);
  cputs("1. Get catalog of disk directory           \n");
  cputs("2. Transfer file from ADAM to MS-DOS format\n");
  cputs("3. Transfer file from MS-DOS to ADAM format\n");
  cputs("4. Get status of file                      \n");
  cputs("5. Rename file                             \n");
  cputs("6. Delete file                             \n");
  cputs("7. Format a disk for ADAM                  \n");
  cputs("8. Copy an entire ADAM disk                \n");
  /* cputs("9. Boot ADAM disk under the ADAM simulator \n");
  cputs("A. Switch to ADAM simulator menu           \n"); */

  cputs("\n");
  ctextbackground(BLUE);
  cputs("Select 1-9 or A or press Q to quit:\n");
  choice = (char) getkey(); fflush(stdin);
  if ('1'<=choice && choice <= '8')
      selectdrive("Which drive (A or B) is the ADAM disk in?\n",&drive);
                        /* get drive */
  switch (choice) {
   case '1': stat = catalog(drive); break;
   case '2': cprintf("Enter name of EOS file (including type):");
             getfname(name);
             cprintf("Enter full path of DOS file to create:");
             getfname(path);
             stat = load(drive,name,path);
             break;
   case '3': cprintf("Enter full path of DOS file to transfer:");
             getfname(path);
             cprintf("Enter name of EOS file to create (including type):");
             getfname(name);
             stat = save(drive,name,path);
             break;
   case '4': cprintf("Enter name of EOS file (including type):");
             getfname(name);
             stat = filestat(drive,name); break;
   case '5': cprintf("Enter old name of EOS file (including type):");
             getfname(name);
             cprintf("Enter new name for EOS file (including type):");
             getfname(newname);
             stat = eosrename(drive,name,newname); break;
   case '6': cprintf("Enter name of EOS file to delete (including type):");
             getfname(name);
             stat = delete(drive,name); break;
   case '7': stat = formatdisk(drive); break;
   case '8': stat = copydisk(drive); break;
   /* case '9': stat = bootdisk(drive); break; */
   /* case 'A': simmain(); break; */
   case 'Q':
   case 'q': exit(0);
  }
  if (!stat) cputs("ERROR in file operation!!!\n");
  cputs("Press RETURN for menu...\n");
  getkey(); fflush(stdin);
 }      /* end while */
}     /* end main program */
