/*****************************************************************************
 * ugBASIC - an isomorphic BASIC language compiler for retrocomputers        *
 *****************************************************************************
 * Copyright 2021-2022 Marco Spedaletti (asimov@mclink.it)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *----------------------------------------------------------------------------
 * Concesso in licenza secondo i termini della Licenza Apache, versione 2.0
 * (la "Licenza"); è proibito usare questo file se non in conformità alla
 * Licenza. Una copia della Licenza è disponibile all'indirizzo:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Se non richiesto dalla legislazione vigente o concordato per iscritto,
 * il software distribuito nei termini della Licenza è distribuito
 * "COSÌ COM'È", SENZA GARANZIE O CONDIZIONI DI ALCUN TIPO, esplicite o
 * implicite. Consultare la Licenza per il testo specifico che regola le
 * autorizzazioni e le limitazioni previste dalla medesima.
 ****************************************************************************/

 /* 6809 optimizations for ugBasic by S.Devulder
  *
  * The idea here is to look for specific patterns over consecutive (1 to 4)
  * lines of code generated by ugBasic compiler, and reorganize it locally
  * by another pattern that contains a more efficent code achieving the same
  * result.
  *
  * It does multiple passes over the source as local rewrites can be
  * applied on previous rewrites, resulting in a avalanche effects that
  * optimizes the source far away from one can guess from the basic patterns.
  *
  * The optimizer can also perform some data flow analysis. It can for
  * instance see that one of the accumulator is $00 to simplify code. It can
  * also detect that some data in memory are written but never read. These
  * are called dead-data. It will then remove all reference to these data in
  * the code making it smaller and faster (as a useless write is not
  * performed anymore).
  *
  * Real full data-flow analysis should normally be able to detect data which
  * is written two times in a row whithout being read in-between. These are
  * also called dead-data since the first written value is also lost. These
  * dead-data typically appear during the avalanche effect occuring during
  * previous optimizations passes. Write operation to these dead-data can
  * also be removed harmlessly.
  *
  * Unfortunately this version of the optimized doesn't have yet a complete
  * data-flow analyzer. Instead, heuristics is used to guess whether a memory
  * operation accesses a dead data. In that case the optimizer will indicate
  * in the commented-out code that the data is *presumed dead*. The
  * heuristics are carefully choosen from the patterns generated by ugBasic
  * and make good guesses. But if you consider these as too dangerous you
  * can disable all of them with the "ALLOW_UNSAFE" flag below.
  *
  * Last the optimizer will also attempt to reorganize the data to get a
  * faster & shorter code. Some data will be "inlined" in the code, making
  * the code auto-modifiable which is no problem for the mc6809. And some
  * "heavily used" data will be moved into the direct-page location for
  * faster accesses.
  */

/****************************************************************************
 * INCLUDE SECTION 
 ****************************************************************************/

#include "../../ugbc.h"
#include <stdarg.h>

/****************************************************************************
 * CODE SECTION 
 ****************************************************************************/

#define DIRECT_PAGE     0x2100
#define LOOK_AHEAD      5
#define ALLOW_UNSAFE    1
#define KEEP_COMMENTS   1

#define DO_DIRECT_PAGE  1
#define DO_INLINE       1
#define DO_UNREAD       1

/* expanable string */
typedef struct {
    char *str; /* actual string */
    int   len; /* string length (not counting null char) */
    int   cap; /* capacity of buffer */
} *buffer;

/* deallocate a buffer */
static buffer buf_del(buffer buf) {
    if(buf != NULL) {
        free(buf->str);
        buf->str = NULL;
        buf->cap = 0;
        buf->len = 0;
        free(buf);
    }

    return NULL;
}

/* allocate a buffer */
static buffer buf_new(int size) {
    buffer buf = malloc(sizeof(*buf));
    if(buf != NULL) {
        buf->len = 0;
        buf->cap = size+1;
        buf->str = malloc(buf->cap);
        buf->str[0] = '\0';
    }
    return buf;
}

/* ensure the buffer can hold len data */
static buffer _buf_cap(buffer buf, int len) {
    if(len+1 >= buf->cap) {
        buf->cap = len + 1 + MAX_TEMPORARY_STORAGE;
        buf->str = realloc(buf->str, buf->cap);
    }
    return buf;
}

/* append a string to a buffer */
static buffer buf_cat(buffer buf, char *string) {
    if(buf != NULL) {
        int len = strlen(string);
        _buf_cap(buf, buf->len + len);
        strcpy(&buf->str[buf->len], string);
        buf->len += len;
    }
    return buf;
}

/* copy a string into a buffer */
static buffer buf_cpy(buffer buf, char *string) {
    if(buf != NULL) buf->len = 0;
    return buf_cat(buf, string);
}

/* append a char at the end of the buffer */
static inline buffer buf_add(buffer buf, char c) {
    if(buf) {
        _buf_cap(buf, buf->len + 1);
        buf->str[buf->len] = c;
        ++buf->len;
        buf->str[buf->len] = '\0';
    }
    return buf;
}

/* vprintf like function */
static buffer buf_vprintf(buffer buf, const char *fmt, va_list ap) {
    if(buf != NULL) {
        int len = 0, avl;
        do {
            _buf_cap(buf, buf->len + len);
            avl = buf->cap - buf->len;
            len = vsnprintf(&buf->str[buf->len], avl, fmt, ap);
        } while(len >= avl);
        buf->len += len;
    }
    return buf;
}

/* sprintf like function */
#ifdef __GNUC__
static buffer buf_printf(buffer buf, const char *fmt, ...)
    __attribute__ ((format (printf, 2, 3)));
#endif
static buffer buf_printf(buffer buf, const char *fmt, ...) {
    va_list ap;
    va_start(ap, fmt);
    buf_vprintf(buf, fmt, ap);
    va_end(ap);
    return buf;
}

/* fgets-like */
static buffer buf_fgets(buffer buf, FILE *f) {
    int c;

    buf_cpy(buf, "");

    while( (c = fgetc(f)) != EOF) {
        buf_add(buf, (char)c);
        if(c=='\n') break;
    }

    return buf;
}

/* strcmp */
static int buf_cmp(buffer a, buffer b) {
    if(a) return b ? strcmp(a->str, b->str) : 1;
    else return -1;
}

#define TMP_BUF_POOL 32
static struct tmp_buf_pool {
    buffer buf;
    void *key1;
    int key2;
} tmp_buf_pool[TMP_BUF_POOL];

/* an integer hash
   https://gist.github.com/badboy/6267743
*/
static unsigned int tmp_buf_hash(unsigned int key) {
    key ^= (key<<17) | (key>>16);
    return key;
}

/* a static one-time buffer */
static buffer tmp_buf(void *key1, unsigned int key2) {
    int hash = tmp_buf_hash(((intptr_t)key1)*31 + key2) % TMP_BUF_POOL;
    struct tmp_buf_pool *tmp = &tmp_buf_pool[hash];
    int count = 0;

    while(tmp->buf!=NULL && (tmp->key1!=key1 || tmp->key2!=key2)) {
        ++count;
        if(++tmp == &tmp_buf_pool[TMP_BUF_POOL]) {
            tmp = tmp_buf_pool;
        }
    }

    if(tmp->buf == NULL) {
        if(count == TMP_BUF_POOL) {
            fprintf(stderr, "TMP_BUF_POOL to short\n");
            exit(-1);
        }
        tmp->buf  = buf_new(0);
        tmp->key1 = key1;
        tmp->key2 = key2;
    }

    return tmp->buf;
}
#define TMP_BUF tmp_buf(__FILE__, __LINE__)

static void tmp_buf_clr(void *key1) {
    struct tmp_buf_pool *tmp = &tmp_buf_pool[0];
    for(;tmp!=&tmp_buf_pool[TMP_BUF_POOL];++tmp) {
        if(tmp->key1 == key1) tmp->buf = buf_del(tmp->buf);
    }
}
#define TMP_BUF_CLR tmp_buf_clr(__FILE__)

/* returns true if the buffer matches a comment or and empty line */
int isAComment( buffer buf ) {
    char * _buffer = buf->str;

    if ( ! *_buffer ) {
        return 1;
    }
    if ( *_buffer == '\r' || *_buffer == '\n' ) {
        return 1;
    }
    while( * _buffer ) {
        if ( *_buffer == ' ' || *_buffer == '\t' ) { 
            ++_buffer;
        } else if ( *_buffer == ';' ) {
            return 1;
        } else {
            return 0;
        }
    }
    return 0;
}

/* returns an UPPER-cased char */
static inline char _toUpper(char a) {
    return (a>='a' && a<='z') ? a-'a'+'A' : a;
}

/* returns true if char is end of line ? */
static inline int _eol(char c) {
    return c=='\0' || c=='\n';
}

/* returns true if both char matches */
static inline int _eq(char pat, char txt) {
    return (pat<=' ') ? (txt<=' ') : (_toUpper(pat)==_toUpper(txt));
}

/* a version of strcmp that ends at EOL and deal our special equality. */
int _strcmp(buffer _s, buffer _t) {
    char *s = _s->str, *t = _t->str;

    while(!_eol(*s) && !_eol(*t) && _eq(*s,*t)) {
        ++s;
        ++t;
    }
    return _eol(*s) && _eol(*t) ? 0 : _eol(*s) ? 1 : -1;
}

/* Matches a string:
    - ' ' maches anthing <= ' ' (eg 'r', \n', '\t' or ' ' )
    - '*' matches up to the next one in the pattern.
   Matched content is copied into buffers passed as varargs. If
   a passed variable is NULL the matched content corresponding
   to it is not copied.

   Returns the last matched '*' or the buffer if pattern is fully
   matched, or NULL otherwise meaning "no match".
*/
static buffer match(buffer _buf, const char *_pattern, ...) {
    buffer ret = _buf;
    const char *s = _buf->str, *p = _pattern;
    va_list ap;

    va_start(ap, _pattern);

    while(!_eol(*s) && *p) {
        if(*p==' ') {while(*p==' ') ++p;
            if(!_eq(' ', *s)) {
                ret = NULL;
                break;
            }
            while(!_eol(*s) && _eq(' ', *s)) ++s;
        } else if(*p=='*') {
            buffer m = va_arg(ap, buffer); ++p;
            if(m != NULL) {
                ret = buf_cpy(m, "");
            }
            while(!_eol(*s) && !_eq(*p, *s)) buf_add(m, *s++);
            if(!_eq(*p,*s)) {
                ret = NULL;
                break;
            }
        } else if(_toUpper(*s++) != _toUpper(*p++)) {
            ret = NULL;
            break;
        }
    }

    va_end(ap);

    return *p=='\0' ? ret : NULL;
}

/* returns true if buf matches any op using the ALU between memory and a register */
static int chg_reg(buffer buf, buffer REG) {
    if(match(buf, " ADD* ", REG)) return 1;
    if(match(buf, " AND* ", REG)) return 1;
    if(match(buf, " CMP* ", REG)) return 1;
    if(match(buf, " EOR* ", REG)) return 1;
    if(match(buf, " LD* ",  REG)) return 1;
    if(match(buf, " OR* ",  REG)) return 1;
    if(match(buf, " SBC* ", REG)) return 1;
    if(match(buf, " SUB* ", REG)) return 1;

    return 0;
}

/* returns true if buf matches an op that sets the CCR */
static int sets_flag(buffer buf, char REG) {
    buffer tmp = TMP_BUF;

    if(chg_reg(buf, tmp)        && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " ASL*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " ASR*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " COM*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " DEC*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " INC*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " LSL*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " LSR*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " ROL*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " ROR*", tmp) && _toUpper(*tmp->str)==REG) return 1;
    if(match(buf, " TST*", tmp) && _toUpper(*tmp->str)==REG) return 1;
	if(match(buf, " ST* ", tmp) && _toUpper(*tmp->str)==REG) return 1;

    return 0;
}

/* returns true if this is a conditionnal branch */
static int isConditionnal(buffer buf) {
   /* short jumps */
    if(match(buf, " BGT ")) return 1;
    if(match(buf, " BLE ")) return 1;
    if(match(buf, " BGE ")) return 1;
    if(match(buf, " BLT ")) return 1;
    if(match(buf, " BEQ ")) return 1;
    if(match(buf, " BNE ")) return 1;
    if(match(buf, " BHI ")) return 1;
    if(match(buf, " BLS ")) return 1;
    if(match(buf, " BHS ")) return 1;
    if(match(buf, " BLO ")) return 1;
    if(match(buf, " BMI ")) return 1;
    if(match(buf, " BPL ")) return 1;

    /* long jumps */
    if(match(buf, " LBGT ")) return 1;
    if(match(buf, " LBLE ")) return 1;
    if(match(buf, " LBGE ")) return 1;
    if(match(buf, " LBLT ")) return 1;
    if(match(buf, " LBEQ ")) return 1;
    if(match(buf, " LBNE ")) return 1;
    if(match(buf, " LBHI ")) return 1;
    if(match(buf, " LBLS ")) return 1;
    if(match(buf, " LBHS ")) return 1;
    if(match(buf, " LBLO ")) return 1;
    if(match(buf, " LBMI ")) return 1;
    if(match(buf, " LBPL ")) return 1;

    return 0;
}

/* returns true if buf matches a jump */
static int isBranch(buffer buf) {
    if(match(buf, " BRA ")) return 1;
    if(match(buf, " BSR ")) return 1;
    if(match(buf, " JMP ")) return 1;
    if(match(buf, " JSR ")) return 1;
	return isConditionnal(buf);
 }

/* number of lines changed */
static int change        = 0;
static int peephole_pass = 0;
static int num_dp        = 0; /* number of variables relocated to direct-page */
static int num_inlined   = 0; /* number of variables inlined */
static int num_unread    = 0; /* number of variables not read */

#ifdef __GNUC__
static void optim(buffer buf, const char *rule, const char *repl, ...)
    __attribute__ ((format (printf, 3, 4)));
#endif

#define R__(X)  #X
#define R_(X)  R__(X)
#define RULE "r" R_(__LINE__) " "

/* replaces the buffer with an optimized code */
/* original buffer is kept as comment */
static void optim(buffer buf, const char *rule, const char *repl, ...) {
    va_list ap;
    buffer tmp = TMP_BUF;
    char *s;

    va_start(ap, repl);
    buf_cpy(tmp, "");

    /* add our own comment if any */
    if(rule) buf_printf(tmp, "; peephole(%d): %s\n", peephole_pass, rule);

    /* comment out line */
    buf_cat(tmp, ";");

    /* copy upto the end of string or upto end of string */
    if ( (s = strchr(buf->str, '\n')) != NULL) *s = '\0'; /* cut at \n */
    buf_cat(tmp, buf->str);
    if( s != NULL ) buf_add(tmp, *s++ = '\n'); /* restore \n */

    /* insert replacement if provided */
    if(repl) {
        buf_vprintf(tmp, repl, ap);
        buf_cat(tmp, "\n");
    }

    /* copy remaining comments */
    if(s) buf_cat(tmp, s);

    /* write result back into input buffer */
    buf_cpy(buf, tmp->str);

    /* one more change */
    ++change;

    va_end(ap);
}

/* returns true if the buffer matches a zero value */
static int isZero(char *s) {
    if(*s == '$') ++s;
    while(*s == '0') ++s;
    return _eq(' ', *s);
}
static int _isZero(buffer buf) {
    return buf!=NULL && isZero(buf->str);
}

/* perform basic peephole optimization with a length-4 look-ahead */
static void basic_peephole(buffer buf[LOOK_AHEAD], int zA, int zB) {
    /* allows presumably safe operations */
    int unsafe = ALLOW_UNSAFE;

    /* various local buffers */
    buffer v1 = TMP_BUF;
    buffer v2 = TMP_BUF;
    buffer v3 = TMP_BUF;
    buffer v4 = TMP_BUF;
    
    /* move B stuff after A stuff */
    if( (match(buf[0], " LDB *", v1) || match(buf[0], " STB *", v1)) && !strchr("AD$", v1->str[0])
    &&  sets_flag(buf[1], 'A') 
	&&  (!match(buf[1], " * *", NULL, v2) || _strcmp(v1, v2)) ) {
        int x = 1, i;
        if( match(buf[x+1], "* equ ", NULL)) ++x;
        if(!match(buf[x+1], " IF ") && !isConditionnal(buf[x+1])) {
            buf_cpy(v1, buf[0]->str);
            for(i=0; i<x; ++i) buf_cpy(buf[i], buf[i+1]->str);
            buf_cpy(buf[x], v1->str);
        }
    }
	/* move D stuff before X stuff */
	if( (match(buf[0], " LDX _*", v1) || match(buf[0], " LDX #*", v1) || match(buf[0], " STX _*", v1)) 
    &&   match(buf[1], " *DD _*", NULL,v2) 
	&&   strchr(v1->str,'+')==NULL
	&&   strchr(v2->str,'+')==NULL
	&&  _strcmp(v1, v2)) {
        int x = 1, i;
        if( match(buf[x+1], "* equ ", NULL)) ++x;
        if(!match(buf[x+1], " IF ") && !isConditionnal(buf[x+1])) {
            buf_cpy(v1, buf[0]->str);
            for(i=0; i<x; ++i) buf_cpy(buf[i], buf[i+1]->str);
            buf_cpy(buf[x], v1->str);
        } else {
			printf("XXX %s", buf[x+1]->str);
		}
    }
	
    
    /* a bunch of rules */
	if( match( buf[0], " LDA #*", v1)
	&&  match( buf[1], " LDB #*", v2)) {
	    optim( buf[0], RULE "(LDA,LDB)->(LDD)", NULL);
		optim( buf[1], NULL, "\tLDD #((%s)&255)*256+((%s)&255)", v1->str, v2->str);
    }
	if ( match( buf[0], " ST* *", v1, v2 )
    &&   match( buf[1], " LD* *", v3, v4 )
    &&  _strcmp(v1, v3)==0
    &&  _strcmp(v2, v4)==0) {
        if(0 && unsafe && match(v2, "_Ttmp") && !match(buf[2], "*SR ") && !(*v1->str=='D' && match(buf[2], " IF "))) {
            char *fmt = NULL;
            /* in case flags are necessary (IF,LBcc), insert TST or LEAX */
            if(match(buf[2], " IF ") && match(buf[3], " LB"))
                fmt = *v1->str=='X' ? "\tLEAX ,X" : "\tTST%c";
            optim( buf[0], "(unsafe, presumed dead)", fmt, _toUpper(*v1->str));
        }

        if(unsafe && match(buf[2], " * [", NULL))
            optim( buf[0], "(unsafe, presumed dead)", NULL);

        if ( strcmp( v2->str, "$A7C1") ) {
            optim( buf[1], RULE "(STORE*,LOAD*)->(STORE*)", NULL);
        }
    }


    if ( match( buf[0], " CLR *", v1 )
    &&   match( buf[2], " ST* *", v2, v3 )
    &&   strchr("AB", _toUpper(*v2->str))
    &&  _strcmp(v1, v3)==0) {
        optim( buf[0], RULE "(CLEAR*,?,STORE*)->(?,STORE*)", NULL);
    }

    if ( _isZero(match(buf[0], " LD* #*", v1, v2) )
    &&   strchr("AB", _toUpper(*v1->str)) ) {
        optim(buf[0], RULE "(LOAD#0)->(CLEAR)", "\tCLR%c", _toUpper(*v1->str));
    }

    if ( match(buf[0], " EOR* #$ff", v1)
    &&   strchr("AB", _toUpper(*v1->str)) ) {
        optim(buf[0], RULE "(EOR#$FF)->(COM)", "\tCOM%c", _toUpper(*v1->str));
    }

    if ( (match(buf[0], " LD* ", v1) || match(buf[0], " CLR*", v1))
    &&   match(buf[1], " LD* ", v2)
    &&  _strcmp(v1,v2)==0) {
        optim(buf[0], RULE "(LOAD/CLR,LOAD)->(LOAD)", NULL);
    }
    
    if ( match( buf[0], " LD")
    &&   match( buf[1], " ST")
    && _strcmp( buf[2], buf[0] )==0
    && !strchr( buf[0]->str, '+' )
    && unsafe) {
        optim( buf[2], RULE "(LOAD*,STORE,LOAD*)->(LOAD*,STORE)", NULL);
    }
    if ( match( buf[0], " LD")
    &&   match( buf[1], " ST")
    &&   match( buf[2], " ST")
    && _strcmp( buf[3], buf[0] )==0
    && unsafe) {
        optim( buf[3], RULE "(LOAD*,STORE,STORE,LOAD*)->(LOAD*,STORE,STORE)", NULL);
    }
    if ( match( buf[0], " LD")
    &&   match( buf[1], " ST")
    &&   match( buf[2], " ST")
    &&   match( buf[3], " ST")
    && _strcmp( buf[4], buf[0] )==0
    && unsafe) {
        optim( buf[4], RULE "(LOAD*,STORE,STORE,STORE,LOAD*)->(LOAD*,STORE,STORE,STORE)", NULL);
    }

    if ( match(buf[0], " ST* *", NULL, v1)
    &&   match(buf[1], " LD* *", NULL, v2)
    &&  ( !strchr(v1->str,'+') && !strchr(v2->str,'+') )
    && _strcmp(v1, v2)!=0
    && _strcmp(buf[0],buf[2])==0 ) {
        optim(buf[0], RULE "(STORE*,LOAD,STORE*)->(LOAD,STORE*)", NULL);
    }
    
    if ( (match( buf[0], " LD* ", v1) || match( buf[0], " CLR*", v1))
    &&   (match( buf[1], " LD* ", v2) || match( buf[1], " CLR*", v2))
    &&  _strcmp( v1, v2)==0) {
        optim(buf[0], RULE "(LOAD/CLR,LOAD/CLR)->(LOAD/CLR)", NULL);
    }


    if ( match(buf[0], " ST")
    && _strcmp(buf[0], buf[1])==0) {
        optim(buf[0], RULE "(STORE*,STORE*)->(STORE*)", NULL);
    }
    if ( (match(buf[0], " ST* *+", NULL, v1) || match(buf[0], " ST* *", NULL, v1))
    &&   !isBranch(buf[1]) && match(buf[1], " * *", NULL, v2) && _strcmp(v1, v2)!=0
	&&  strchr(buf[0]->str,'+')==NULL
	&&  strchr(buf[1]->str,'+')==NULL
    && _strcmp(buf[2], buf[0])==0) {
        optim(buf[0], RULE "(STORE*,?,STORE*)->(?,STORE*)", NULL);
    }
    if ((match(buf[0], " ST* *+", NULL, v1) || match(buf[0], " ST* *", NULL, v1))
    &&  !isBranch(buf[1]) && match(buf[1], " * *", NULL, v2) && _strcmp(v1, v2)!=0
    &&  !isBranch(buf[2]) && match(buf[2], " * *", NULL, v2) && _strcmp(v1, v2)!=0
    && _strcmp(buf[3], buf[0])==0) {
        optim(buf[0], RULE "(STORE*,?,?,STORE*)->(?,?,STORE*)", NULL);
    }

    if( (match(buf[0], " LD* ", v1) || match(buf[0], " ST* ",v1))
    && _isZero(match(buf[1], " CMP* #*", v2, v3))
    && _strcmp(v1, v2)==0) {
        optim(buf[1], RULE "(LOAD/STORE,CMP#0)->(LOAD/STORE)", NULL);
    }

    if ( match(buf[0], " LDD *", v1)
    &&   match(buf[1], " STD _Ttmp*", v2)
    &&   match(buf[2], " CLRB")
    &&   match(buf[3], " LDX *", v3)
    &&   match(buf[4], " CMPX _Ttmp*", v4)
    &&  _strcmp(v2, v4)==0) {
        if(unsafe) {
            optim(buf[0], RULE "(LDD+,STD*,LDX,CMPX*)->(LDX,CMP+)", NULL);
            optim(buf[1], "(unsafe, presumed dead)", NULL);
            optim(buf[4], NULL, "\tCMPX %s", v1->str);
        } else {
            optim(buf[4], RULE "(LDD+,STD*,LDX,CMPX*)->(LDD+,STD*,LDX,CMPX+)", "\tCMPX %s", v1->str);
        }
    }

    if ( match(buf[0], " LDD *", v1)
    &&   match(buf[1], " STD _Ttmp*", v2)
    &&   match(buf[2], " LDD *", v3)
    &&   match(buf[3], " ADD _Ttmp*", v4)
    &&  _strcmp(v2, v4)==0) {
        if(unsafe) {
            optim(buf[0], RULE "(LDD+,STD*,LDD,ADDD*)->(LDD,ADD+)", NULL);
            optim(buf[1], "(unsafe, presumed dead)", NULL);
            optim(buf[3], NULL, "\tADDD %s", v1->str);
        } else {
            optim(buf[3], RULE "(LDD+,STD*,LDD,ADDD*)->(LDD+,STD*,LDD,ADD+)", "\tADDD %s", v1->str);
        }
    }

    if ( match(buf[0], " STD *", v1)
    &&   match(buf[1], " LDX *", v2)
    &&  _strcmp(v1,v2)==0) {
        //if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
        optim(buf[1], RULE "(STD*,LDX*)->(STD*,TDX)", "\tTFR D,X");
    }

    if ( match(buf[0], " STD *", v1)
    &&   match(buf[1], " LDB *+1", v2)
    &&  _strcmp(v1, v2)==0) {
        if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
        optim(buf[1], RULE "(STD,LDB+1)->()", NULL);
    }

    if ( match(buf[0], " LDD #*", v1)
    &&   match(buf[1], " ADDD #*", v2) 
    &&  !match(buf[2], "* equ ", NULL)) {
        optim(buf[0], RULE "(LDD#,ADD#)->(LDD#)", "\tLDD #%s+%s", v1->str, v2->str);
        optim(buf[1], NULL, NULL);
    }

    if ( match(buf[0], " STX *", v1)
    &&   match(buf[1], " CLRA")
    &&   match(buf[2], " LDX *", v2)
    &&  _strcmp(v1,v2)==0) {
        optim(buf[0], RULE "(STX*,CLRA,LDX*)->(CLRA,STX*)", NULL);
        optim(buf[2], NULL, "\tSTX %s", v1->str);
    }

    if ( match(buf[0], " STD *", v1)
    &&   match(buf[1], " LDD *", v2)
    &&   match(buf[2], " ADDD *", v3)
    &&  _strcmp(v1,v3)==0) {
        // if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
        optim(buf[1], RULE "(STD*,LDD+,ADD*)->(STD*,ADD+)", NULL);
        optim(buf[2], NULL, "\tADDD %s", v2->str);
    }

    if ( match(buf[0], " STB *", v1)
    &&   match(buf[1], " LDB *", v2)
    &&   match(buf[2], " *B *", v3, v4)
    &&  _strcmp(v1,v4)==0
    &&   (match(v3, "OR") || match(v3,"AND") || match(v3,"EOR") || match(v3,"ADD"))
    ) {
        if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
        optim(buf[1], RULE "(STB*,LDB+,ORB/ANDB/EORB/ADDB*)->(STB*,ORB/ANDB/EORB/ADDB+)", NULL);
        optim(buf[2], NULL, "\t%sB %s", v3->str, v2->str);
    }


    if ( match(buf[0], " STD _Ttmp*", v1)  // 6
    &&   match(buf[1], " LD* [_Ttmp*]", v2, v3) // 9
    &&  _strcmp(v1,v3)==0) {
        //if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
        optim(buf[1], RULE "(STD,LDD[])->(TDX,LOAD*X)", "\tTFR D,X\n\tLD%c ,X", _toUpper(*v2->str));
    }

    if ( match(buf[1], " TST*", v1)
    &&  sets_flag(buf[0], *v1->str)) {
        optim(buf[1], RULE "(FLAG-SET,TST)->(FLAG-SET)", NULL);
    }

    if ( match(buf[0], " LDB #$01")
    &&   match(buf[1], " LDX *", v1)
    &&   match(buf[2], " JSR CPUMATHMUL16BITTO32*", NULL)) {
        optim(buf[0], RULE "(MUL#1)->(NOP)", "\tLDD %s", v1->str);
        optim(buf[1], NULL, "\tLDX #0");
        optim(buf[2], NULL, NULL);
    }

    // A VOIR
    if ( match(buf[0], " STB *", v1)
    &&   match(buf[1], " STA ")
    &&   match(buf[2], " LDA *", v2)
    &&   match(buf[3], " STA *", v3)
    &&  _strcmp(v1, v2)==0
    &&   unsafe /* A is considered dead after */) {
        optim(buf[2], RULE "(STB*,STA,LDA*,STA+)->(STB*,STA,STB+)", NULL);
        optim(buf[3], NULL, "\tSTB %s", v3->str);
    }

    if( match(buf[0], " STD _Ttmp*", v1)
    &&  match(buf[1], " LDD _Ttmp*", v2)
    &&  match(buf[2], " LDX _Ttmp*", v3)
    && _strcmp(v1,v3)==0
    && !match(buf[3], " IF ")
    &&  unsafe) {
        optim(buf[0], RULE "(STD*,LDD,LDX*)->(TDX,LDD)", "\tTFR D,X");
        optim(buf[2], NULL, NULL);
    }

    if( match(buf[0], " LDD *", v1)
    &&  match(buf[1], " STD *", v2)
    &&  match(buf[2], " TFR D,X")
    && !match(buf[3], " *SR ", NULL)) {
        optim(buf[0], RULE "(LDD*,STD+,TDX)->(LDX*,STX+)", "\tLDX %s", v1->str);
        optim(buf[1], NULL, "\tSTX %s", v2->str);
        optim(buf[2], NULL, NULL);
    }

    if( match(buf[0], " STD *", v1)
    &&  match(buf[1], " ST* *", NULL, v2)
    &&  match(buf[2], " LDD *", v3)
    && _strcmp(v1, v3)==0
    && _strcmp(v1, v2)!=0
    &&  strchr((const char*)v2 ,'+')==NULL) {
        optim(buf[2], RULE "(STD*,ST?,LDD*)->(STD*,ST?)", NULL);
    }
    
    if( match(buf[0], " STD *", v1)
    &&  match(buf[1], " LSLB")
    &&  match(buf[2], " ROLA")
    &&  match(buf[3], " STD *", v2)
    && _strcmp(v1, v2)==0) {
        optim(buf[0], RULE "(STD*,LSLB,ROLA,STD*)->(LSLB,ROLA,STD*)", NULL);
    }

    if( match(buf[0], " STD *", v1)
    &&  match(buf[1], " ADDD *", v2)
    &&  match(buf[2], " STD *", v3)
    && _strcmp(v1, v3)==0
    && _strcmp(v1, v2)!=0) {
        optim(buf[0], RULE "(STD*,ADDD,STD*)->(ADDD,STD*)", NULL);
    }
    
    if ( peephole_pass>2
    &&   match(buf[0], " STD _Ttmp*", v1)
    &&   match(buf[1], " LD* ", v2)
    &&   match(buf[2], " ST* [_Ttmp*]", v3, v4)
    &&  _strcmp(v2,v3)==0
    &&  _strcmp(v1,v4)==0) {
        if(unsafe)
            optim(buf[0], RULE "(unsafe, presumed dead) (STD,LOAD,STORE[])->(TDX,LOAD,STORE*X)", "\tTFR D,X");
        else
            optim(buf[0], RULE "(STD,LOAD,STORE[])", "\tSTD _Ttmp%s\n\tTFR D,X", v1->str);
        optim(buf[2], NULL, "\tST%c ,X", *v2->str);
    }
	
	if(match( buf[0], " STD *", v1)
	&& match( buf[1], " LDB *+1", v2)
	&& _strcmp(v1, v2)==0) {
		optim(buf[1], RULE "(STD,LDB+1)", NULL);
	}
	if(match( buf[0], " STD *", v1)
	&& match( buf[1], " ST* *", NULL, v2) && 0!=_strcmp(v1,v2)
	&& match( buf[2], " LDB *+1", v2)
	&& _strcmp(v1, v2)==0) {
		optim(buf[2], RULE "(STD,?,LDB+1)", NULL);
	}
	if(match( buf[0], " STD *", v1)
	&& match( buf[1], " ST* *", NULL, v2) && 0!=_strcmp(v1,v2)
	&& match( buf[2], " ST* *", NULL, v2) && 0!=_strcmp(v1,v2)
	&& match( buf[3], " LDB *+1", v2)
	&& _strcmp(v1, v2)==0) {
		optim(buf[3], RULE "(STD,?,?,LDB+1)", NULL);
	}
}

/* check if buffer matches any of xxyy (used for LDD #$xxyy op) */
static buffer chkLDD(buffer buf, char *xxyy, buffer value) {
    return match( buf, " LDD #$*", value) &&
        value->len==4 &&
        (xxyy[0]=='-' || xxyy[0]==value->str[0]) &&
        (xxyy[1]=='-' || xxyy[1]==value->str[1]) &&
        (xxyy[2]=='-' || xxyy[2]==value->str[2]) &&
        (xxyy[3]=='-' || xxyy[3]==value->str[3]) ? buf : NULL;
}

/* can this opcode make A non zero */
static int can_nzA(buffer buf) {
    char *s;

    for(s = buf->str; !_eol(*s) && *s!=','; ++s);

    if(!match(buf, " ")) return 1;
    if(match(buf, " ADDA ")) return 1;
    if(match(buf, " ADDD ")) return 1;
    if(match(buf, " BSR ")) return 1;
    if(match(buf, " COMA")) return 1;
    if(match(buf, " DECA")) return 1;
    if(match(buf, " EORA ")) return 1;
    if(match(buf, " EXG ")) return 1;
    if(match(buf, " INCA")) return 1;
    if(match(buf, " JSR ")) return 1;
    if(match(buf, " LDA ")) return 1;
    if(match(buf, " LDD ")) return 1;
    if(match(buf, " ORA ")) return 1;
    if(match(buf, " PULS ")) return 1;
    if(match(buf, " PULU ")) return 1;
    if(match(buf, " ROLA")) return 1;
    if(match(buf, " RORA")) return 1;
    if(match(buf, " RTI")) return 1;
    if(match(buf, " RTS")) return 1;
    if(match(buf, " SBCA ")) return 1;
    if(match(buf, " SEX")) return 1;
    if(match(buf, " SUBA ")) return 1;
    if(match(buf, " SUBD ")) return 1;
    if(match(buf, " TFR ") && s[0]==',' && (s[1]=='A' || s[1]=='D')) return 1;

    return 0;
}

/* can this opcode make B non zero */
static int can_nzB(buffer buf) {
    char *s;

    for(s = buf->str; !_eol(*s) && *s!=','; ++s);

    if(!match(buf, " ")) return 1;
    if(match(buf, " ADDB ")) return 1;
    if(match(buf, " ADDD ")) return 1;
    if(match(buf, " BSR ")) return 1;
    if(match(buf, " COMB")) return 1;
    if(match(buf, " DECB")) return 1;
    if(match(buf, " EORB ")) return 1;
    if(match(buf, " EXG ")) return 1;
    if(match(buf, " INCB")) return 1;
    if(match(buf, " JSR ")) return 1;
    if(match(buf, " LDB ")) return 1;
    if(match(buf, " LDD ")) return 1;
    if(match(buf, " ORB ")) return 1;
    if(match(buf, " PULS ")) return 1;
    if(match(buf, " PULU ")) return 1;
    if(match(buf, " ROLB")) return 1;
    if(match(buf, " RORB")) return 1;
    if(match(buf, " RTI")) return 1;
    if(match(buf, " RTS")) return 1;
    if(match(buf, " SBCB ")) return 1;
    if(match(buf, " SUBB ")) return 1;
    if(match(buf, " SUBD ")) return 1;
    if(match(buf, " TFR ") && s[0]==',' && (s[1]=='B' || s[1]=='D')) return 1;

    return 0;
}

/* optimizations related to A or B being zero */
static void optim_zAB(buffer buf[LOOK_AHEAD], int *zA, int *zB) {
    buffer v1 = TMP_BUF;
    buffer v2 = TMP_BUF;
    buffer v3 = TMP_BUF;

    int unsafe = ALLOW_UNSAFE;

    if(*zA) {
        if (match( buf[0], " CLRA")) {
            optim( buf[0], RULE "[A=0](CLRA)->()", NULL);
        } else if (match( buf[0], " LDA #$ff")) {
            optim( buf[0], RULE "[A=0](LDA#ff)->(DECA)", "\tDECA");
            *zA = 0;
        } else if ( match(buf[0], " LDA #$01")) {
            optim( buf[0], RULE "[A=0](LDA#1)->(INCA)", "\tINCA");
            *zA = 0;
        } else if ( chkLDD( buf[0], "00--", v1)) {
            optim(buf[0], RULE "[A=0](LDD#00xx)->(LDB#xx)", "\tLDB #$%c%c", v1->str[2], v1->str[3]);
            *zB = 0;
        } else if (match( buf[0], " TFR A,B")) {
            optim( buf[0], RULE "[A=0](TAB)->(CLRB)", "\tCLRB");
            *zB = 1;
        } else if (peephole_pass>2 
               &&  match( buf[0], " STA *", v1)
               &&  match( buf[1], " LDB *", v2)
               &&  _strcmp(v1,v2)==0) {
            optim(buf[1], RULE "[A=0](STA*,LDB*)->(CLRB)", "\tCLRB");
        } else if (*zB
               &&  match(buf[0], " ADDD *", v1)) {
            optim(buf[0], RULE "[D=0](ADD)->(LDD)", "\tLDD %s", v1->str);
        } else if (*zB
               &&  match(buf[0], " STD _Ttmp*", v1)
               &&  match(buf[1], " LDX *", v2)
               &&  match(buf[2], " CMPX _Ttmp*", v3)
               && _strcmp(v1, v3)==0) {
            if(unsafe) optim(buf[0], "(unsafe, presumed dead)", NULL);
            optim(buf[2], RULE "[D=0](STD*,LDX,CMPX*)->(LDX)", NULL);
        } else if(can_nzA(buf[0])) {
            *zA = 0;
        }
    } else if ( chkLDD(buf[0], "00--", v1) || match( buf[0], " LDD #0") || match( buf[0], " CLRA") ) {
        *zA = 1;
    }

    if(*zB) {
        if (match( buf[0], " CLRB")) {
            optim( buf[0], RULE "[B=0](CLRB)->()", NULL);
        } else if (match( buf[0], " LDB #$ff")) {
            optim( buf[0], RULE "[B=0](LDB#ff)->(DECB)", "\tDECB");
            *zB = 0;
        } else if (match( buf[0], " LDB #$01")) {
            optim( buf[0], RULE "[B=0](LDB#1)->(INCB)", "\tINCB");
            *zB = 0;
        } else if ( chkLDD( buf[0], "--00", v1) ) {
            optim( buf[0], RULE "[B=0](LDB#xx00)->(LDA#xx)", "\tLDA #$%c%c", v1->str[0], v1->str[1]);
            *zA = 0;
        } else if (match( buf[0], " TFR B,A")) {
            optim( buf[0], RULE "[B=0](TBA)->(CLRA)", "\tCLRA");
            *zA = 1;
        } else if(can_nzB(buf[0])) {
            *zB = 0;
        }
    } else if ( chkLDD(buf[0], "--00", v1) || match( buf[0], " LDD #0") || match( buf[0], " CLRB") ) {
        *zB = 1;
    }
    
    if(!*zA
    && match( buf[0], " LDB #$*",  v1)
    && match( buf[1], " STB ")
    && match( buf[2], " CLRA")) {
        optim(buf[0], RULE "(LDB#,STB,CLRA)->(LDD#,STB)", "\tLDD #$00%s", v1->str);
        optim(buf[2], NULL, NULL);
        *zA = 0;
    }
}

/* optimizations related to variables */

/* variables database */
static struct {
    struct var {
        char *name;
#define NO_REORG  1
#define NO_DP     2
#define NO_INLINE 4
#define NO_REMOVE 8
        int flags;
        int size;
        int nb_rd;
        int nb_wr;
        int offset; /* 0=unchanged, >0 offset to page 0; -1 = candidate for inlining, -2 = inlined */
        char *init;
    } *tab;
    int capacity;
    int size;
    int page0_max;
} vars;

/* clears the database */
static void vars_clear(void) {
    int i;
    for(i=0; i<vars.size; ++i) {
        struct var *v = &vars.tab[i];
        free(v->name);
        if(v->init) free(v->init);
    }
    vars.size = 0;
    vars.page0_max = 0;
}

/* gets (or creates) an entry for a variable from the data-base */
struct var *vars_get(buffer _name) {
    char *name = _name->str;
    struct var *ret = NULL;
    int i;

    char *s=strchr(name,'+');
    if(s) *s='\0';

    for(i=0; i<vars.size ; ++i) {
        if(strcmp(vars.tab[i].name, name)==0) {
            ret = &vars.tab[i];
        }
    }
    if(ret == NULL) {
        if(vars.size == vars.capacity) {
            vars.capacity += 16;
            vars.tab = realloc(vars.tab, sizeof(*vars.tab)*vars.capacity);
        }
        ret = &vars.tab[vars.size++];
        ret->name   = strdup(name);
        ret->flags  = 0;
        ret->size   = 0;
        ret->nb_rd  = 0;
        ret->nb_wr  = 0;
        ret->offset = 0;
        ret->init   = NULL;
    }
    if(s) *s='+';

    return ret;
}

static int vars_ok(buffer name) {
    if(match(name, "_Tstr"))   return 0;
    if(match(name, "_label"))  return 0;

    if(name->str[0]=='_')      return 1;
    if(match(name, "CLIP"))    return 1;
    if(match(name, "XCUR"))    return 1;
    if(match(name, "YCUR"))    return 1;
    if(match(name, "CURRENT")) return 1;
    if(match(name, "FONT"))    return 1;
    if(match(name, "TEXT"))    return 1;
    if(match(name, "LAST"))    return 1;
    if(match(name, "XGR"))     return 1;
    if(match(name, "YGR"))     return 1;
    if(match(name, "FREE_"))   return 1;

    return 0;
}

/* look for variable uses and collect data about he variables */
static void vars_scan(buffer buf[LOOK_AHEAD]) {
    buffer tmp = TMP_BUF;
    buffer arg = TMP_BUF;

    // if( match( buf[0], " * _*+", NULL, buf) ) {
        // struct var *v = vars_get(buf);
        // v->flags |= NO_INLINE;
    // }

    if( match( buf[0], " * #*",  NULL, arg)
    ||  match( buf[0], " * [*]", NULL, arg) ) if(vars_ok(arg)) {
        struct var *v = vars_get(arg);
        v->flags |= NO_REMOVE/*|NO_DP*/;
        v->nb_rd++;
    }

    if( match( buf[0], " CLR *",  arg)
    ||  match( buf[0], " ST* *",  tmp, arg) ) if(vars_ok(arg)) {
        struct var *v = vars_get(arg);
        v->nb_wr++;
    }

    if (match( buf[0], " ADD* *", NULL, arg)
    ||  match( buf[0], " ADC* *", NULL, arg)
    ||  match( buf[0], " AND* *", NULL, arg)
    ||  match( buf[0], " CMP* *", NULL, arg)
    ||  match( buf[0], " EOR* *", NULL, arg)
    ||  match( buf[0], " LD* *",  NULL, arg)
    ||  match( buf[0], " OR* *",  NULL, arg)
    ||  match( buf[0], " SBC* *", NULL, arg)
    ||  match( buf[0], " SUB* *", NULL, arg) ) if(vars_ok(arg)) {
        struct var *v = vars_get(arg);
        v->nb_rd++;
    }
    if( match( buf[0], " ASL *", arg)
    ||  match( buf[0], " ASR *", arg)
    ||  match( buf[0], " COM *", arg)
    ||  match( buf[0], " DEC *", arg)
    ||  match( buf[0], " INC *", arg)
    ||  match( buf[0], " LSL *", arg)
    ||  match( buf[0], " LSR *", arg)
    ||  match( buf[0], " ROL *", arg)
    ||  match( buf[0], " ROR *", arg)
    ||  match( buf[0], " TST *", arg)) if(vars_ok(arg)) {
        struct var *v = vars_get(arg);
        v->nb_wr++;
        v->nb_rd++;
    }
    if( match(buf[0], " * *",   tmp, arg)
    ||  match(buf[0], " * [*]", tmp, arg) ) if(vars_ok(arg)) {
        struct var *v = vars_get(arg);
        v->offset = -1; /* candidate for inlining */
    }

    if( match( buf[0], "* rzb *", tmp, arg) && vars_ok(tmp)) {
        struct var *v = vars_get(tmp);
        v->size = atoi(arg->str);
        v->init = strdup("1-1");
    }

    if( match(buf[0], "* fcb *", tmp, arg) && vars_ok(tmp) && strchr(buf[0]->str,',')==NULL) {
        struct var *v = vars_get(tmp);
        v->size = 1;
        v->init = strdup(isZero(arg->str) ? "1-1" : arg->str);
    }

    if( match(buf[0], "* fdb *", tmp, arg) && vars_ok(tmp) && strchr(buf[0]->str,',')==NULL) {
        struct var *v = vars_get(tmp);
        v->size = 2;
        v->init = strdup(arg->str);
    }

    /* heurstic to find the max used index in direct-page */
    if( match(buf[0], "* equ $*", tmp, arg)
    &&  arg->len==2
    &&  (*tmp->str!='_' || tmp->str[1]=='T')) {
        int v = strtol(arg->str, NULL, 16);
        if (v >= vars.page0_max) vars.page0_max = v+2;
    }
}

/* compares two variables according to their access-count */
static int vars_cmp(const void *_a, const void *_b) {
    const struct var *a = _a;
    const struct var *b = _b;

    int diff = ((a->nb_rd + a->nb_wr) - (b->nb_rd + b->nb_wr));

    return -(diff!=0 ? diff : strcmp(a->name, b->name)); // Ttmp < Tstr
}

/* decide which variable goes in direct-page, which will be inlined */
static void vars_prepare_relocation(void) {
    int i;

    num_dp      = 0;
    num_inlined = 0;

    qsort(vars.tab, vars.size, sizeof(*vars.tab), vars_cmp);

    for(i = 0; i<vars.size; ++i) {
        struct var *v = &vars.tab[i];

        /* skip over unknown size var  */
        if(v->size == 0)  continue;

        /* skip over unread variables */
        if(v->nb_rd == 0) continue;

        /* flagged as not inline */
        if(v->flags & NO_INLINE) v->offset = 0;
        if(!DO_INLINE)           v->offset = 0;

        /* can't inline > 2 bytes */
        if(v->offset == -1 && v->size>2) v->offset = 0;

        /* check if inlining is good */
        if(v->offset == -1) {
            /* LDA: imm=2, dp=4, extended=5
               LDD: imm=3, dp=5, extended=6 */
            int dp_cost     = (3+v->size)*(v->nb_rd + v->nb_wr);
            int inline_cost = (4+v->size)*(v->nb_rd + v->nb_wr - 1)+(1+v->size);
            
            if( (v->init==NULL || isZero(v->init) || 0==strcmp("1-1", v->init)) &&  dp_cost < inline_cost ) {
                if(DO_DIRECT_PAGE) v->offset = 0;
            } 
        }
        if(DO_DIRECT_PAGE 
        && 0==v->offset
        && 0==(v->flags && NO_DP)
        && v->size<=4 /* not too big to let room for others */
        && vars.page0_max + v->size <= 256
        ) {
            v->offset = vars.page0_max;
            vars.page0_max += v->size;
        }

        // printf("%s %d/%d %d %d %d\n", v->name, v->nb_rd, v->nb_wr, v->offset, v->size, vars.page0_max);
    }
}

/* removes unread variables */
static void vars_remove(buffer buf[LOOK_AHEAD]) {
    buffer var = TMP_BUF;
    buffer op  = TMP_BUF;
    
    if(!DO_UNREAD) return;
    
    /* unread */
    if(match( buf[0], " ST* *", op, var) && vars_ok(var)) {
        struct var *v = vars_get(var);
        if(v->nb_rd == 0 && v->offset!=-2) {
            char *rep = NULL;
            v->offset = 0;
            if(match(buf[1], " IF ") && match(buf[2], " LB")) {
                if(*op->str=='X') rep = "\tLEAX ,X";
                else {
                    static char tst[8];
                    sprintf(tst, "\tTST%c", *op->str);
                    rep = tst;
                }
            }
            optim(buf[0], "unread", rep != NULL ? "%s" : NULL, rep);
        }
    }

    /* remove changed variables */
    if(match( buf[0], "* rzb ", var)
    || match( buf[0], "* fcb ", var)
    || match( buf[0], "* fdb ", var) ) if(vars_ok(var)) {
        struct var *v = vars_get(var);
        if(v->nb_rd==0 && 0<v->size && v->size<=4 && 0==(v->flags & NO_REMOVE) && v->offset!=-2) {
            optim(buf[0], "unread",NULL);
            ++num_unread;
        }             
     }
}            


/* performs optimizations related to variables relocation */
static void vars_relocate(buffer buf[LOOK_AHEAD]) {
    buffer REG = TMP_BUF;
    buffer var = TMP_BUF;
    buffer op  = TMP_BUF;
    
    /* direct page or inlined */
   if(match( buf[0], " * *", op, var) && vars_ok(var) ) {
        struct var *v = vars_get(var);
        if(v->offset > 0) {
            optim(buf[0], "direct-page", "\t%s <%s", op->str, var->str);
        } else if(v->offset == -1 && chg_reg(buf[0], REG)
               && ((strchr("DXYU", *REG->str)!=NULL  && v->size==2) || v->size==1) ) {
            v->offset = -2;
            v->flags |= NO_REMOVE;
            optim(buf[0], "inlined", "\t%s #%s%s\n%s equ *-%d", op->str,
                  v->init==NULL ? "*" : v->init,
                  v->init==NULL ? (v->size==2 ? "" : "&255") : "",
                  var->str, v->size);
        }            
    }

    if(match( buf[0], " * [*]", op, var) && vars_ok(var)) {
        struct var *v = vars_get(var);
        if(v->offset > 0) {
            optim(buf[0], "direct-page", "\t%s [%s+$%04x]", op->str, var->str, DIRECT_PAGE);
        } else if(v->offset == -1 && strstr(var->str,"+$")==NULL) {
            v->offset = -2;
            optim(buf[0], "inlined", "\t%s >%s\n%s equ *-2", op->str,
                v->init==NULL ? var->str : v->init, var->str);
            }            
        }

    if(match( buf[0], " * #*", op, var) && vars_ok(var) ) {
        struct var *v = vars_get(var);
        if(v->offset > 0 && strstr(var->str,"+$")==NULL) {
            optim(buf[0], "direct-page", "\t%s #%s+$%04x", op->str, var->str, DIRECT_PAGE);
        }            
    }
    
    /* remove changed variables */
    if(match( buf[0], "* rzb ", var)
    || match( buf[0], "* fcb ", var)
    || match( buf[0], "* fdb ", var) ) if(vars_ok(var)) {
        struct var *v = vars_get(var);
        if(v->offset > 0) {
            optim(buf[0], "direct-page", "%s equ $%02x", var->str, v->offset);
            ++num_dp;
        } else if(v->offset == -2) {
            optim(buf[0], "inlined", NULL);
            ++num_inlined;
         }            
     }
}            

/* collapse all heading spaces into a single tabulation */
static void out(FILE *f, buffer _buf) {
    char *s = _buf->str;
    int tab = 0;
    while(*s==' ' || *s=='\t') {tab = 1; ++s;}
    if(tab) fputs("\t", f);
    fputs(s, f);
}

/* remove space that is sometimes used in indexing mode and makes the optimized produce bad dcode */
static void fixes_indexed_syntax(buffer buf) {
    char *s = buf->str;

    /* not an instruction */
    if(!_eq(' ', *s)) return;

    /* skip over spaces */
    do ++s; while(*s && _eq(' ', *s));

    /* comment */
    if(*s==';') return;

    /* skip over instruction */
    while(*s && !_eq(' ', *s)) ++s;
    if(!*s) return;

    /* skip over spaces */
    do ++s; while(*s && _eq(' ', *s));
    if(!*s) return;

    /* process argment */
    do ++s; while(*s && !_eq(' ', *s));
    if(!*s) return;

    /* space found check if case "LDA 1, X" */
    if(s[-1]==',') {
        char *t = s;
        do ++t; while(*t && _eq(' ', *t));
        switch(_toUpper(*t)) {
        case 'X': case 'Y': case 'U': case 'S':
            /* yes ==> move register after coma */
            *s = *t;
            *t = ' ';
            break;
        default:
            break;
        }            
    }
}

/* various kind of optimization */
enum OPT_KIND {PEEPHOLE, DEADVARS, RELOCATION1, RELOCATION2};
static int optim_pass( Environment * _environment, buffer buf[LOOK_AHEAD], enum OPT_KIND kind) {
    char fileNameOptimized[MAX_TEMPORARY_STORAGE];
    FILE * fileAsm;
    FILE * fileOptimized;
    int i;
    int still_to_go = LOOK_AHEAD;

    int line = 0;
    int zA = 0, zB = 0;

    sprintf( fileNameOptimized, "%s.asm", get_temporary_filename( _environment ) );
        
    /* prepare for phase */
    switch(kind) {
        case DEADVARS:
        ++peephole_pass;
        num_unread  = 0;
        break;
        
        case RELOCATION1:
        ++peephole_pass;
        vars_prepare_relocation();
        break;
        
        case RELOCATION2:
        break;
        
        case PEEPHOLE:
        ++peephole_pass;
        vars_clear();
        break;
    }

    fileAsm = fopen( _environment->asmFileName, "rt" );
    if(fileAsm == NULL) {
        perror(_environment->asmFileName);
        exit(-1);
    }

    fileOptimized = fopen( fileNameOptimized, "wt" );
    if(fileOptimized == NULL) {
        perror(fileNameOptimized);
        exit(-1);
    }            
    
    /* clears our look-ahead buffers */
    for(i = 0; i<LOOK_AHEAD; ++i) buf_cpy(buf[i], "");

    /* global change flag */
    change = 0;

    while( still_to_go ) {
        /* print out oldest buffer */
        if ( line >= LOOK_AHEAD ) out(fileOptimized, buf[0]);

        /* shift the buffers */
        for(i=0; i<LOOK_AHEAD-1; ++i) buf_cpy(buf[i], buf[i+1]->str);

        /* read next line, merging adjacent comments */
        if(feof(fileAsm)) {
            --still_to_go;
            buf_cpy(buf[LOOK_AHEAD-1], "");
        } else do {
            /* read next line */
            buf_fgets( buf[LOOK_AHEAD-1], fileAsm );
            fixes_indexed_syntax(buf[LOOK_AHEAD-1]);
            /* merge comment with previous line if we do not overflow the buffer */
            if(isAComment(buf[LOOK_AHEAD-1])) {
                if(KEEP_COMMENTS) buf_cat(buf[LOOK_AHEAD-2], buf[LOOK_AHEAD-1]->str);
                buf_cpy(buf[LOOK_AHEAD-1], "");
            } else break;
        } while(!feof(fileAsm));

        switch(kind) {
            case PEEPHOLE:
            basic_peephole(buf, zA, zB);
            optim_zAB(buf, &zA, &zB);
            
            /* only look fo variable when no peephole has been performed */
            if(change == 0) vars_scan(buf);
            break;
            
            case DEADVARS:
            vars_remove(buf);
            break;
            
            case RELOCATION1:
            case RELOCATION2:
            vars_relocate(buf);
            break;
        }

        ++line;
    }

    /* log info at the end of the file */
    switch(kind) {
        case PEEPHOLE:
        fprintf(fileOptimized, "; peephole: pass %d, %d change%s.\n", peephole_pass, 
            change, change>1 ?"s":"");
        break;
        
        case DEADVARS:
        fprintf(fileOptimized, "; peephole: pass %d, %d var%s removed.\n", peephole_pass, 
            num_unread, num_unread>1 ?"s":"");
        break;
        
        case RELOCATION2:
        fprintf(fileOptimized, "; peephole: pass %d, %d var%s moved to dp, %d var%s inlined.\n", peephole_pass, 
            num_dp, num_dp>1 ?"s":"", 
            num_inlined, num_inlined>1 ? "s":"");
        break;
        
        default:
        break;
    }
    
    (void)fclose(fileAsm);
    (void)fclose(fileOptimized);

    /* makes our generated file the new asm file */
    remove(_environment->asmFileName);
    (void)rename( fileNameOptimized, _environment->asmFileName );
    
    return change;
}

/* main entry-point for this service */
void target_peephole_optimizer( Environment * _environment ) {
    if ( _environment->peepholeOptimizationLimit > 0 ) {
        buffer buf[LOOK_AHEAD];
        int i;

        for(i=0; i<LOOK_AHEAD; ++i) buf[i] = buf_new(0);

        int optimization_limit_count = _environment->peepholeOptimizationLimit;

        do {
            while(optim_pass(_environment, buf, PEEPHOLE)&&optimization_limit_count) {
                --optimization_limit_count;
            };
            optim_pass(_environment, buf, DEADVARS);
        } while(change&&optimization_limit_count);
        optim_pass(_environment, buf, RELOCATION1);
        optim_pass(_environment, buf, RELOCATION2);

        for(i=0; i<LOOK_AHEAD; ++i) buf[i] = buf_del(buf[i]);
        TMP_BUF_CLR;
    }
}
