	'
	' Sound effect editor for SN76489 and AY-3-8910
	'
	' by Oscar Toledo G.
	' http://nanochess.org/
	'
	' Creation date: May/07/2021.
	' Revision date: Oct/31/2024. Ported to CVBasic
	' Revision date: Nov/02/2024. Port completed.
	'

	ON FRAME GOSUB play_effects

	CONST SN76489 = 1
	CONST AY38910 = 2

	CONST SOUND_LENGTH = 20
	CONST DEBOUNCE_TIME = 12
	CONST WINDOW_HEIGHT = 8
	CONST DEFAULT_FREQUENCY = 170
	CONST DEFAULT_VOLUME = 12

	CONST #MENU_BASE = 545
	CONST #HEADER_BASE = 64
	CONST #HEADER2 = 416
	CONST #NOTES_BASE = 128

	DIM #freq(SOUND_LENGTH)
	DIM vol(SOUND_LENGTH)
	DIM noise(SOUND_LENGTH)
	DIM mix(SOUND_LENGTH)
	
	CLS
	MODE 2
	WAIT

	FOR #c = $0400 TO $07FF
		VPOKE #c, VPEEK(#c - $0400)
	NEXT #c

	FOR #c = $2000 TO $200F
		VPOKE #c, $31
	NEXT #c

	VPOKE $2006, $81
	VPOKE $2007, $81

	FOR #c = $2010 TO $201F
		VPOKE #c, $f6
	NEXT #c

	VPOKE $2016, $a6
	VPOKE $2017, $a6

	GOSUB show_title

	PRINT AT 65,"Select the mode:"

	PRINT AT 129,"1- SN76489"
	PRINT AT 161,"   (Colecovision/SG1000/Others)"
	PRINT AT 225,"2- AY-3-8910"
	PRINT AT 257,"   (MSX/SVI/Nabu/Einstein)"

	PRINT AT 481, "For use with CVBasic"
	PRINT AT 513, "Developed by Oscar Toledo G."

	PRINT AT 737, "https://nanochess.org"
	#c = 736
	GOSUB reverse_video

	debounce = DEBOUNCE_TIME
	DO
		WAIT
		IF debounce THEN debounce = debounce - 1
		c = CONT.KEY
		IF debounce THEN c = 15
	LOOP WHILE c <> 1 AND c <> 2
	sound_chip = c

	debounce = DEBOUNCE_TIME
	
	window = 0
	index = 0
	#freq(0) = DEFAULT_FREQUENCY
	vol(0) = DEFAULT_VOLUME
	length = 1
	speed = 12
	repeat = 0

	GOSUB show_headers
	GOSUB show_menu

	'
	' Main loop
	'
main_loop:
	WAIT
	IF debounce THEN debounce = debounce - 1
	GOSUB highlight_item
	PRINT AT #HEADER_BASE + 1,"Index:",<2>index

	'
	' Move highlight with disc
	'
	c = CONT
	d = c AND $e0
	IF (d = $00) AND debounce = 0 THEN
		IF CONT.UP THEN	' Up
			GOSUB dehighlight_item
			IF index > 0 THEN index = index - 1
			IF index < window THEN
				window = window - 1
				GOSUB update_list
			END IF
			debounce = DEBOUNCE_TIME
		END IF
		IF CONT.DOWN THEN	' Down
			GOSUB dehighlight_item
			IF index < length - 1 THEN index = index + 1
			IF index >= window + WINDOW_HEIGHT THEN
				window = window + 1
				GOSUB update_list
			END IF
			debounce = DEBOUNCE_TIME
		END IF
	END IF

	c = CONT.key
	IF debounce THEN c = 15
	IF c = 1 THEN	' 1- Enter frequency
		debounce = DEBOUNCE_TIME
		GOSUB enter_frequency
		IF #value <> -1 THEN
			IF sound_chip = SN76489 THEN
				if #value > 1023 THEN #value = 1023
			ELSE
				if #value > 4095 THEN #value = 4095
			END IF
			#freq(index) = #value
		END IF
		GOSUB update_item
	ELSEIF c = 2 THEN	' 2- Enter volume
		debounce = DEBOUNCE_TIME
		GOSUB enter_volume
		IF #value <> -1 THEN
			IF #value > 15 THEN #value = 15
			vol(index) = #value
		END IF
		GOSUB update_item
	ELSEIF c = 3 THEN	' 3- Enter noise
		debounce = DEBOUNCE_TIME
		GOSUB enter_noise
		IF #value <> -1 THEN
			IF sound_chip = SN76489 THEN
				IF #value <> 0 THEN	
					IF #value < 8 THEN #value = 8
					IF #value > 15 THEN #value = 15
				END IF
			ELSE
				IF #value > 31 THEN #value = 31
			END IF
			noise(index) = #value
		END IF
		GOSUB update_item
	ELSEIF c = 4 THEN	' 4- Toggle mix of noise/tone
		debounce = DEBOUNCE_TIME
		IF sound_chip = SN76489 THEN
			GOSUB enter_mix
			IF #value <> -1 THEN
				IF #value > 15 THEN #value = 15
				mix(index) = #value
			END IF
		ELSE
			mix(index) = mix(index) XOR 1
		END IF
		GOSUB update_item
	ELSEIF c = 5 THEN	' 5- Increase length of sound effect
		GOSUB dehighlight_item
		debounce = DEBOUNCE_TIME
		IF length < SOUND_LENGTH THEN
			#freq(length) = DEFAULT_FREQUENCY
			vol(length) = DEFAULT_VOLUME
			length = length + 1
		END IF
		WAIT
		GOSUB update_list
		GOSUB show_length
	ELSEIF c = 6 THEN	' 6- Decrease length of sound effect
		GOSUB dehighlight_item
		debounce = DEBOUNCE_TIME
		IF length > 1 THEN length = length - 1
		IF index >= length THEN index = length - 1
		IF window >= length - WINDOW_HEIGHT THEN
			IF length - WINDOW_HEIGHT < 0 THEN
				window = 0
			ELSE	
				window = length - WINDOW_HEIGHT
			END IF
		END IF
		WAIT
		GOSUB update_list
		GOSUB show_length
	ELSEIF c = 7 THEN	' 7- Enter speed for playing (video frames for each tone)
		debounce = DEBOUNCE_TIME
		GOSUB enter_speed
		IF #value <> -1 THEN
			IF #value < 1 THEN #value = 1
			IF #value > 12 THEN #value = 12
			speed = #value
		END IF
		GOSUB show_speed
	ELSEIF c = 8 THEN	' 8- Toggle repeat when playing
		debounce = DEBOUNCE_TIME
		repeat = repeat XOR 1
		GOSUB show_repeat
	ELSEIF c = 9 THEN	' 9- Play sound effect
		debounce = DEBOUNCE_TIME
		sound_effect = sound_effect XOR 1
		sound_state = 0
	ELSEIF c = 0 THEN	' 0- Show source code
		debounce = DEBOUNCE_TIME
		GOSUB show_source
		GOSUB show_headers
		GOSUB show_menu
	END IF

	GOTO main_loop

	'
	' Show menu
	'
show_menu:	PROCEDURE
	PRINT AT #MENU_BASE,"1-Edit frequency"
	PRINT AT #MENU_BASE + 32,"2-Edit volume"
	PRINT AT #MENU_BASE + 64,"3-Edit noise"
	IF sound_chip = SN76489 THEN
		PRINT AT #MENU_BASE + 96,"4-Noise vol."
	ELSE
		PRINT AT #MENU_BASE + 96,"4-Toggle noise"
	END IF
	PRINT AT #MENU_BASE + 128,"5-Incr. length"
	PRINT AT #MENU_BASE + 16,"6-Decr. length"
	PRINT AT #MENU_BASE + 48,"7-Edit speed"
	PRINT AT #MENU_BASE + 80,"8-Toggle repeat"
	PRINT AT #MENU_BASE + 112,"9-Play"
	PRINT AT #MENU_BASE + 144,"0-Show source"
	PRINT AT #MENU_BASE + 192,"Move up and down with joystick"
	END

	'
	' Show title
	'
show_title: 	PROCEDURE
	CLS
	IF sound_chip = SN76489 THEN
		PRINT AT 2,"Sound effect editor SN76489"
	ELSEIF sound_chip = AY38910 THEN
		PRINT AT 1,"Sound effect editor AY-3-8910"
	ELSE
		PRINT AT 6,"Sound effect editor"
	END IF
	#c = 0: GOSUB reverse_video

	END

	'
	' Show headers
	'
show_headers:	PROCEDURE
	GOSUB show_title
	GOSUB show_length
	GOSUB show_speed
	GOSUB show_repeat
	GOSUB update_list
	END

	'
	' Show source code
	'
show_source:	PROCEDURE
	CLS
	#c = 0
	d = 0
	IF sound_chip = SN76489 THEN
		WHILE d < length
			PRINT AT #c,"SOUND 2,",<>#freq(d),",",<>vol(d)
			#c = #c + 32
			PRINT AT #c,"SOUND 3,"
			IF noise(d) THEN PRINT <>noise(d)
			PRINT ",",<>mix(d)
			#c = #c + 32
			d = d + 1
			IF d = length THEN EXIT WHILE
			IF #c = 736 THEN
				PRINT AT 737,"Press key to continue"
				DO
					WAIT
					IF debounce THEN debounce = debounce - 1
					c = CONT.KEY
					IF debounce THEN c = 15
				LOOP WHILE c = 15
				debounce = DEBOUNCE_TIME
				CLS
				#c = 0
			END IF
		WEND
	ELSE
		WHILE d < length
			PRINT AT #c,"SOUND 2,",<>#freq(d),",",<>vol(d)
			#c = #c + 32
			PRINT AT #c,"SOUND 4,"
			IF mix(d) THEN PRINT <>noise(d)
			PRINT ",$",<>3 - mix(d) * 2,"8"
			#c = #c + 32
			d = d + 1
			IF d = length THEN EXIT WHILE
			IF #c = 736 THEN
				PRINT AT 737,"Press key to continue"
				DO
					WAIT
					IF debounce THEN debounce = debounce - 1
					c = CONT.KEY
					IF debounce THEN c = 15
				LOOP WHILE c = 15
				debounce = DEBOUNCE_TIME
				CLS
				#c = 0
			END IF
		WEND
	END IF

	PRINT AT 737,"Press key to exit"
	DO
		WAIT
		IF debounce THEN debounce = debounce - 1
		c = CONT.KEY
		IF debounce THEN c = 15
	LOOP WHILE c = 15
	debounce = DEBOUNCE_TIME

	END

show_length:	PROCEDURE
	PRINT AT #HEADER2 + 1,"Length:",<2>length
	END

show_speed:	PROCEDURE
	PRINT AT #HEADER_BASE + 11,"Speed:",<2>speed
	END

show_repeat:	PROCEDURE
	PRINT AT #HEADER_BASE + 20,"Repeat:"
	IF repeat THEN PRINT "Yes" ELSE PRINT "No "
	END

enter_speed:	PROCEDURE
	#c = #HEADER_BASE + 17
	d = 2
	GOSUB read_number
	END

enter_frequency:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE + 5
	d = 4
	GOSUB read_number
	END

enter_volume:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE + 14
	d = 2
	GOSUB read_number
	END

enter_noise:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE + 23
	d = 2
	GOSUB read_number
	END

enter_mix:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE + 30
	d = 2
	GOSUB read_number
	END

read_number:	PROCEDURE
	PRINT AT #c
	FOR e = 0 TO d - 1
		VPOKE $1800 + #c + e, " " + 128
	NEXT e
	
	#value = 0
	DO
		VPOKE $1800 + #c, "_" + 128

		DO
			WAIT
			IF debounce THEN debounce = debounce - 1
			e = CONT.key
			IF debounce THEN e = 15
		LOOP WHILE e = 15

		debounce = DEBOUNCE_TIME

		IF e = 10 THEN #value = -1: RETURN

		IF e < 10 THEN
			#value = #value * 10 + cont.key
			VPOKE $1800 + #c, cont.key + 48 + 128
			#c = #c + 1
			d = d - 1
		END IF

	LOOP WHILE d <> 0 AND cont.key <> 11

	END

update_item:	PROCEDURE
	c = index
	d = index - window
	PRINT AT d * 32 + #NOTES_BASE
	GOSUB update_data
	END

update_data:	PROCEDURE
	PRINT "Tone:",<4>#freq(c)
	PRINT " Vol:",<2>vol(c)
	PRINT " Noise:",<2>noise(c)
	IF sound_chip = SN76489 THEN
		PRINT " Vol:",<2>mix(c)
	ELSE
		PRINT " Mix:",<2>mix(c)
	END IF
	END

update_list:	PROCEDURE
	FOR d = 0 TO 7
		c = window + d
		PRINT AT d * 32 + #NOTES_BASE
		IF c >= length THEN
			PRINT "        "
			PRINT "        "
			PRINT "        "
			PRINT "        "
		ELSE
			GOSUB update_data
		END IF
	NEXT d

	END

highlight_item:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE
	FOR c = 0 TO 31
		VPOKE $1800 + #c + c, VPEEK($1800 + #c + c) OR $80
	NEXT c
	END

dehighlight_item:	PROCEDURE
	#c = (index - window) * 32 + #NOTES_BASE
	FOR c = 0 TO 31
		VPOKE $1800 + #c + c, VPEEK($1800 + #c + c) AND $7F
	NEXT c
	END

reverse_video:	PROCEDURE
	FOR c = 0 TO 31
		VPOKE $1800 + #c + c, VPEEK($1800 + #c + c) XOR $80
	NEXT c
	END

	'
	' Play sound effect
	'
play_effects:	PROCEDURE
	ON sound_effect GOSUB sound_none, sound_user
	END

sound_none:	PROCEDURE
	IF sound_chip = SN76489 THEN
		SOUND 2,,0
		SOUND 3,,0
	ELSE
		SOUND 7,,0
		SOUND 9,,$38
	END IF
	END

sound_user:	PROCEDURE
	sound_temp = sound_state / speed
	IF sound_chip = SN76489 THEN
		SOUND 2,#freq(sound_temp),vol(sound_temp)
		IF noise(sound_temp) THEN SOUND 3,noise(sound_temp)
		SOUND 3,,mix(sound_temp)
	ELSE
		SOUND 7,#freq(sound_temp),vol(sound_temp)
		SOUND 9,noise(sound_temp),$38 - mix(sound_temp) * $20
	END IF
	sound_state = sound_state + 1
	IF sound_state = length * speed THEN
		IF repeat THEN
			sound_state = 0
		ELSE
			sound_effect = 0
		END IF
	END IF
	END
