 package PSG_Emu;

import javax.sound.sampled.*;
//import java.lang.Math.*;

/**
 * Speaker - To store and hear a sound stream
 *
 * @author Daniel Bienvenu
 * @version 2005-06-03
 */
public class Speaker implements Runnable
{
    private byte[] i_buffer_sample = new byte[65536];
    private int i_buffer_sample_size;
    private int i_buffer_sample_pointer;
    
    private AudioFormat af_sample;
    private DataLine.Info dl_info;
    private SourceDataLine sdl_line;
    
    private byte current_state;
    
    private final byte S_NOTSET = 0x00;
    private final byte S_READY = 0x01;
    private final byte S_LOADING = 0x02;
    private final byte S_PLAYING = 0x04;
    private final byte S_STOPING = 0x08;
    private final byte S_STOPED = 0x10;
    private final byte S_CLOSING = 0x20;
    private final byte S_CLOSED = 0x40;
    
    public Speaker()
    {
        current_state = S_NOTSET;
    }
    
    public void set_framerate(float f_sample_rate)
    {
        if (current_state != S_NOTSET)
        {
            stop_playing();
            close_dataline();
        }
        af_sample = new AudioFormat(f_sample_rate,8,1,true,false);
        open_dataline();
    }

    public void clean_stream()
    {
        if ( current_state == S_NOTSET) return;
        if ( current_state == S_PLAYING)
        {
            current_state = S_STOPING;
            while (current_state == S_STOPING);
            current_state = S_LOADING;
        }
        i_buffer_sample_size = 0;
    }

    public void add_data(byte data)
    {
        if ( current_state == S_NOTSET) return;
        if ( current_state == S_PLAYING)
        {
            current_state = S_STOPING;
            while (current_state == S_STOPING);
            current_state = S_LOADING;
        }
        if ((i_buffer_sample_size+1) == i_buffer_sample.length)
        {
            byte[] newtable = new byte[i_buffer_sample.length+65536];
            System.arraycopy(i_buffer_sample,0,newtable,0,i_buffer_sample.length);
            i_buffer_sample = newtable;
        }
        i_buffer_sample[i_buffer_sample_size++] = data;
    }
    
    public void stop_playing()
    {
        if ( current_state == S_NOTSET) return;
        current_state = S_STOPING;
        while(current_state == S_STOPING);
        System.gc();
    }
    
    public void start_playing()
    {
        if ( current_state == S_NOTSET) return;
        if ( current_state == S_PLAYING)
        {
            stop_playing();
        }
        current_state = S_PLAYING;
        i_buffer_sample_pointer = 0;
        sdl_line.start();
    }
    
    public void open_dataline()
    {
        try
        {
            dl_info = new DataLine.Info(SourceDataLine.class,af_sample);
            if ( !AudioSystem.isLineSupported( dl_info ) )
            {
                current_state = S_NOTSET;
                return;
            }
            sdl_line = (SourceDataLine) AudioSystem.getLine( dl_info );
            /* the af_sample param is needed here
             * to support well the different versions of Java
             * from 1.4 to 1.5
             */
            sdl_line.open(af_sample); 
            current_state = S_READY;
        }
        catch (Exception e)
        {
            // do Nothing
        }
    }
    
    public void close_dataline()
    {
        try
        {
            sdl_line.stop();
            sdl_line.close();
        }
        catch (Exception e)
        {
            // do Nothing
        }
        current_state = S_CLOSED;
    }
    
    public void kill()
    {
        if ( current_state == S_NOTSET) return;
        if ( current_state == S_PLAYING)
        {
            stop_playing();
        }
        current_state = S_CLOSING;
    }
    
    public void run()
    {
        int data_left, block_size;
        while(current_state != S_CLOSING)
        {
            try
            {
                // Sleep for a while
                Thread.currentThread().sleep(50);
            }
            catch(InterruptedException e)
            {
                // Do nothing
            }
            while(current_state == S_PLAYING)
            {
                data_left = i_buffer_sample_size - i_buffer_sample_pointer;
                block_size = sdl_line.available();
                if (data_left < block_size) block_size = data_left;
                try
                {
                    sdl_line.write(i_buffer_sample,i_buffer_sample_pointer,block_size);
                    try
                    {
                        // Sleep for a while
                        Thread.currentThread().sleep(2);
                    }
                    catch(InterruptedException e)
                    {
                        // Do nothing
                    }
                    i_buffer_sample_pointer += block_size;
                    if (i_buffer_sample_size == i_buffer_sample_pointer)
                    {
                        sdl_line.drain();
                        current_state = S_STOPING;
                    }
                }
                catch (Exception e)
                {
                    // Do Nothing
                }
            }
            if (current_state == S_STOPING)
            {
                try
                {
                    sdl_line.flush();
                    sdl_line.stop();
                }
                catch (Exception e)
                {
                    // Do Nothing
                }
                current_state = S_READY;
            }
        }
        close_dataline();
    }
}
