 package PSG_Emu;

import	java.util.Arrays;
import java.lang.Math.*;

/**
 * PSG_Chip emulate the sound chip inside the ColecoVision, SG1000 and MSX-1 computers.
 *
 * @author Daniel Bienvenu
 * @version 2005-05-13
 */
public class PSG_Chip
{
    // output data line
    Speaker sp;

    // downsampling buffer
    private float[] f_downsampling_buffer = new float[32];
    private int i_downsampling_buffer_pointer;

    // output stream data
    /* Input Clock Frequency = 3,579,545.45 Hz (+/- 10Hz)
     * NTSC = 59.94Hz
     * PAL = 50 Hz
     */

    /*
     * ONESECOND = Number of clock ticks in one second
     *           = (Input Clock Freq / 16)
     *
     * NTSC_FRAME = Number of clock ticks in one NTSC frame
     *              (Input Clock Freq / 16 / NTSC)
     *
     * PAL_FRAME = Number of clock ticks in one PAL frame
     *             (Input Clock Freq / 16 / PAL)
     */
                                    /*  223721 * 16 = 3579536 (not bad) */
    private final int ONESECOND = 223721; /* (int) 223721.5791 */
    private final int NTSC_FRAME = 3732;  /* (int) 3732.4254 */
    private final int PAL_FRAME = 4474;   /* (int) 4474.4316 */
    private float amplitude[] = new float[32];
    private float actual_output_height;
    private float actual_output_center;
    public float output_stream[] = new float[PAL_FRAME]; /* because PAL_FRAME > NTSC_FRAME */

    private int down_sampling;

    // instance chip variables
    private int period[] = new int[4];
    private byte attenuation[] = new byte[4];
    private int counter[] = new int[4];
    private boolean flipflop[] = new boolean[4];
    private int ShiftRegister;

    private int ShiftRegister_PRESET = 0x4000;
    /* This is the initial register value.
     * It should be 0x4000 = %0100 0000 0000 0000.
     */

    private byte noise_code;
    private boolean freq3;
    private boolean white_noise; /// added

    private final byte max_attenuation = 15;

    private final int NoiseFreqs[] = {32,64,128,0};

    /**
     * Constructor for objects of class PSG_Chip
     */
    public PSG_Chip()
    {
        /* initialise instance variables */
        initialize_amplitude_table();
        initialize_chip();
        sp = new Speaker();
        down_sampling = 6;
        sp.set_framerate((float) (ONESECOND / down_sampling) );
        new Thread(sp).start();
    }

    private void initialize_amplitude_table()
    {
        int i;
        amplitude[0] = 1.0f;
        for (i=1;i<15;i++) { amplitude[i] = amplitude[i-1] / 1.258925412f; }
        amplitude[15] = 0.0f;
    }

    public void initialize_chip()
    {
        int i;
        for (i=0;i<4;i++)
        {
            period[i] = 0;
            attenuation[i] = 15;
            counter[i] = 0;
            flipflop[i] = false;
        }
        freq3 = false;
        white_noise = true; /// added
        noise_code = -1;
        actual_output_height = 0.0f;
        actual_output_center = 0.0f;
        ShiftRegister = ShiftRegister_PRESET;
    }

    private void update_noise_period()
    {
        if (freq3)
        {
            period[0] = period[3] * 2;
        }
    }
    
    public void clear_stream()
    {
        int i;
        sp.clean_stream();
    }

    public void start_playing()
    {
        sp.start_playing();
    }

    public void stop_playing()
    {
        sp.stop_playing();
    }

    public void kill()
    {
        sp.kill();
    }

    /**
     * Set MUTE for a specific channel
     *
     * @param  channel   tone channel ( 1 - 3 )
     */
    public void mute(byte channel)
    {
        attenuation[channel] = max_attenuation;
    }

    /**
     * Turn off sound. Set MUTE for all channels
     * 
     */
    public void mute()
    {
        attenuation[0] = max_attenuation;
        attenuation[1] = max_attenuation;
        attenuation[2] = max_attenuation;
        attenuation[3] = max_attenuation;
    }

    /**
     * Set a tone channel values
     *
     * @param  channel   tone channel ( 1 - 3 )
     * @param  new_period   period for the tone channel
     * @param  new_attenuation   attenuation for the tone channel 0 = volume max, 15 = mute.
     */
    public void tone(byte channel, int new_period, byte new_attenuation)
    {
        attenuation[channel] = new_attenuation;
        period[channel] = new_period;
    }

    /**
     * Set the noise channel values
     *
     * @param  code   this code give the frequency (preset or from channel3) and the noise filter (white or periodic)
     * @param  new_attenuation   attenuation for the noise channel 0 = volume max, 15 = mute.
     */
    public void noise(byte code, byte new_attenuation)
    {
        attenuation[0] = new_attenuation;
        
        /* THE COLECO BIOS CHECK BEFORE SENDING TWICE THE SAME NOISE CTRL VALUE */
        if (code != noise_code)
        {
            noise_code = code;
            /*
             * Bit 3 is 1 for white noise; the ShiftRegister is reset every time.
             * Replacement:
             */
            ShiftRegister = ShiftRegister_PRESET;
        }    

            white_noise = ((code & 4) == 4);
        
            code &= 3;

            if (code == 3)
            {
                freq3 = true;
            }
            else
            {
                freq3 = false;
                period[0] = NoiseFreqs[code];
            }
        
    }

    /**
     * tick is an element of the frame rate (not the down sampled one)
     * 
     * @return     the relative amplitude at this time : between -1.0f and 1.0f
     */
    private float tick()
    {
        int ch;
        int feedback; /// added
        float accumulator = 0.0f;

        update_noise_period();

        for (ch = 0 ; ch < 4 ; ch++)
        {
            if (attenuation[ch] == 15)
            {
                counter[ch] = 0;
            }
            else
            {
                if (period[ch]!=0) counter[ch]++;
                if (counter[ch] >= period[ch])
                {
                    counter[ch] = 0;
                    if (ch == 0)
                    {
                        /* NOISE */
                        /*
                         * get output (bit 0, about to be shifted out of the ShiftRegister)
                         */
                        flipflop[0] = ((ShiftRegister & 1) == 1);
                        if (white_noise)
                        {
                            /*
                             * white noise: feedback = bit 0 XOR bit 1
                             */
                            feedback = (ShiftRegister ^ (ShiftRegister >>> 1)) & 1; 
                        }
                        else
                        {
                            /*
                             * periodic noise: feedback = bit 0
                             */
                            feedback = ShiftRegister & 1;
                        }

                        /*
                         * feedback goes into bit 14 as bit 0 is shifted off
                         * >>> fills with zero bits
                         * >> fills with the sign bit (sign extension) - unwanted!
                         * << fills with zero bits
                         */
                        ShiftRegister = (ShiftRegister >>> 1) | (feedback << 14);
                    }
                    else
                    {
                        /* TONE */
                        flipflop[ch] = !flipflop[ch];
                    }
                }
                if (flipflop[ch])
                {
                    accumulator -= amplitude[attenuation[ch]];
                }
            }
        }
        return (accumulator);
    }

    /**
     * refresh proceed a single stream sequence during a refresh (NTSC or PAL) 
     * 
     * @param  isNTSC   true for NTSC rendering, false for PAL.
     * @return     the number of data in the output stream
     */
    public int refresh(boolean isNTSC)
    {
        if (isNTSC)
        {
            dorefresh(NTSC_FRAME);
            return NTSC_FRAME;
        }
        dorefresh(PAL_FRAME);
        return PAL_FRAME;
    }

    private void dorefresh(int number_of_ticks)
    {
        // filters
        float delta;
        int i;
        for (i=0;i<number_of_ticks;i++)
        {
            output_stream[i] = tick();
            // add filter smooth attack
            output_stream[i] = (output_stream[i] + actual_output_height * 15.0f) / 16.0f;
            // update filter smooth attack            
            actual_output_height = output_stream[i];
            // add filter natural center
            output_stream[i] += actual_output_center;
            // update filter natural center
            delta = -output_stream[i] / 360.0f;
            actual_output_center += delta;
        }
        downsampling(number_of_ticks);
    }
    
    private void downsampling(int length_buffer)
    {
        int i,j,result;
        float accumulator = 0;
        
        for(i=0;i<length_buffer;i++)
        {
            f_downsampling_buffer[i_downsampling_buffer_pointer++] = output_stream[i];
            if (i_downsampling_buffer_pointer==down_sampling)
            {
                accumulator = 0;
                for (j=0;j<down_sampling;j++)
                {
                    accumulator += f_downsampling_buffer[j];
                }
                result = Math.round((accumulator * 91.5f) / (down_sampling * 1.0f));
                //result = Math.round((accumulator * 63.75f) / (down_sampling * 1.0f));
                if (result<-128) result = -128;
                if (result>127) result = 127;
                sp.add_data((byte) (result & 0xff));
                i_downsampling_buffer_pointer = 0;
            }
        }
    }

    
}
